const db = require('../models');
const Agent = db.agents;
const Payment = db.payments;
const { Op } = require('sequelize');

// Create and Save a new Agent
exports.create = async (req, res) => {
  try {
    // Validate request
    if (!req.body.name || !req.body.phone) {
      return res.status(400).send({
        message: "Agent name and phone cannot be empty!"
      });
    }

    // Create an Agent
    const agent = {
      name: req.body.name,
      company: req.body.company,
      email: req.body.email,
      phone: req.body.phone,
      address: req.body.address,
      status: req.body.status || 'active'
    };

    // Save Agent in the database
    const data = await Agent.create(agent);
    res.status(201).send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the Agent."
    });
  }
};

// Retrieve all Agents from the database
exports.findAll = async (req, res) => {
  try {
    const { name, status } = req.query;
    let condition = {};

    if (name) {
      condition.name = { [Op.like]: `%${name}%` };
    }
    if (status) {
      condition.status = status;
    }

    const data = await Agent.findAll({ where: condition });
    res.send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving agents."
    });
  }
};

// Find a single Agent with an id
exports.findOne = async (req, res) => {
  try {
    const id = req.params.id;
    const data = await Agent.findByPk(id);
    
    if (data) {
      res.send(data);
    } else {
      res.status(404).send({
        message: `Cannot find Agent with id=${id}.`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error retrieving Agent with id=${req.params.id}`
    });
  }
};

// Update an Agent by the id in the request
exports.update = async (req, res) => {
  try {
    const id = req.params.id;
    const [num] = await Agent.update(req.body, {
      where: { id: id }
    });

    if (num === 1) {
      res.send({
        message: "Agent was updated successfully."
      });
    } else {
      res.send({
        message: `Cannot update Agent with id=${id}. Maybe Agent was not found or req.body is empty!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error updating Agent with id=${req.params.id}`
    });
  }
};

// Delete an Agent with the specified id
exports.delete = async (req, res) => {
  try {
    const id = req.params.id;
    const num = await Agent.destroy({
      where: { id: id }
    });

    if (num === 1) {
      res.send({
        message: "Agent was deleted successfully!"
      });
    } else {
      res.send({
        message: `Cannot delete Agent with id=${id}. Maybe Agent was not found!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Could not delete Agent with id=${req.params.id}`
    });
  }
};

// Get agent payments
exports.getPayments = async (req, res) => {
  try {
    const id = req.params.id;
    const data = await Payment.findAll({
      where: { agentId: id },
      order: [['createdAt', 'DESC']]
    });
    
    res.send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving payments."
    });
  }
};

// Add a payment for an agent
exports.addPayment = async (req, res) => {
  try {
    // Validate request
    if (!req.body.amount || !req.body.type) {
      return res.status(400).send({
        message: "Payment amount and type cannot be empty!"
      });
    }

    const agentId = req.params.id;
    
    // Check if agent exists
    const agent = await Agent.findByPk(agentId);
    if (!agent) {
      return res.status(404).send({
        message: `Agent with id=${agentId} not found!`
      });
    }

    // Start a transaction
    const t = await db.sequelize.transaction();

    try {
      // Create a Payment
      const payment = {
        amount: req.body.amount,
        type: req.body.type,
        method: req.body.method || 'cash',
        reference: req.body.reference,
        notes: req.body.notes,
        agentId: agentId,
        userId: req.body.userId
      };

      // Save Payment in the database
      const newPayment = await Payment.create(payment, { transaction: t });

      // Update agent balance
      let newBalance = parseFloat(agent.balance);
      if (payment.type === 'payment') {
        newBalance -= parseFloat(payment.amount);
      } else {
        newBalance += parseFloat(payment.amount);
      }

      await Agent.update(
        { 
          balance: newBalance,
          totalTransactions: agent.totalTransactions + 1
        },
        { 
          where: { id: agentId },
          transaction: t
        }
      );

      // Commit the transaction
      await t.commit();

      res.status(201).send({
        ...newPayment.dataValues,
        newBalance
      });
    } catch (error) {
      // Rollback the transaction in case of error
      await t.rollback();
      throw error;
    }
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the Payment."
    });
  }
};