const db = require("../models");
const BkashAccountNumber = db.bkashAccountNumbers;
const { Op } = require("sequelize");

// Get all account numbers by type
exports.getAccountNumbersByType = async (req, res) => {
  try {
    const { accountType } = req.params;
    
    const accountNumbers = await BkashAccountNumber.findAll({
      where: { accountType: accountType },
      order: [['isPrimary', 'DESC'], ['createdAt', 'ASC']]
    });

    res.json(accountNumbers);
  } catch (error) {
    console.error('Error getting account numbers:', error);
    res.status(500).json({ 
      message: "Error retrieving account numbers",
      error: error.message 
    });
  }
};

// Get all account numbers
exports.getAllAccountNumbers = async (req, res) => {
  try {
    const accountNumbers = await BkashAccountNumber.findAll({
      order: [['accountType', 'ASC'], ['isPrimary', 'DESC'], ['createdAt', 'ASC']]
    });

    res.json(accountNumbers);
  } catch (error) {
    console.error('Error getting all account numbers:', error);
    res.status(500).json({ 
      message: "Error retrieving all account numbers",
      error: error.message 
    });
  }
};

// Add new account number
exports.addAccountNumber = async (req, res) => {
  try {
    const { accountType, accountNumber, accountName, isPrimary = false } = req.body;

    // Validate account type
    const validTypes = ['agent', 'personal', 'merchant'];
    if (!validTypes.includes(accountType)) {
      return res.status(400).json({ 
        message: "Invalid account type. Must be 'agent', 'personal', or 'merchant'" 
      });
    }

    // Check if account number already exists
    const existingAccount = await BkashAccountNumber.findOne({
      where: { accountNumber: accountNumber }
    });

    if (existingAccount) {
      return res.status(400).json({ 
        message: "Account number already exists" 
      });
    }

    // If this is set as primary, remove primary status from other accounts of same type
    if (isPrimary) {
      await BkashAccountNumber.update(
        { isPrimary: false },
        { where: { accountType: accountType, isPrimary: true } }
      );
    }

    const newAccountNumber = await BkashAccountNumber.create({
      accountType,
      accountNumber,
      accountName: accountName || `${accountType} Account`,
      isPrimary: isPrimary || false,
      isActive: true
    });

    res.status(201).json(newAccountNumber);
  } catch (error) {
    console.error('Error adding account number:', error);
    res.status(500).json({ 
      message: "Error adding account number",
      error: error.message 
    });
  }
};

// Update account number
exports.updateAccountNumber = async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    const accountNumber = await BkashAccountNumber.findByPk(id);
    if (!accountNumber) {
      return res.status(404).json({ message: "Account number not found" });
    }

    // If setting as primary, remove primary status from other accounts of same type
    if (updateData.isPrimary === true) {
      await BkashAccountNumber.update(
        { isPrimary: false },
        { 
          where: { 
            accountType: accountNumber.accountType, 
            isPrimary: true,
            id: { [Op.ne]: id }
          } 
        }
      );
    }

    await accountNumber.update(updateData);
    
    const updatedAccountNumber = await BkashAccountNumber.findByPk(id);
    res.json(updatedAccountNumber);
  } catch (error) {
    console.error('Error updating account number:', error);
    res.status(500).json({ 
      message: "Error updating account number",
      error: error.message 
    });
  }
};

// Delete account number
exports.deleteAccountNumber = async (req, res) => {
  try {
    const { id } = req.params;
    
    const accountNumber = await BkashAccountNumber.findByPk(id);
    if (!accountNumber) {
      return res.status(404).json({ message: "Account number not found" });
    }

    // Check if this account number is being used in transactions
    const BkashTransaction = db.bkashTransactions;
    const transactionCount = await BkashTransaction.count({
      where: { accountNumberId: id }
    });

    if (transactionCount > 0) {
      return res.status(400).json({ 
        message: "Cannot delete account number that has associated transactions. Please deactivate instead." 
      });
    }

    await accountNumber.destroy();
    res.json({ message: "Account number deleted successfully" });
  } catch (error) {
    console.error('Error deleting account number:', error);
    res.status(500).json({ 
      message: "Error deleting account number",
      error: error.message 
    });
  }
};

// Set primary account number
exports.setPrimaryAccount = async (req, res) => {
  try {
    const { id } = req.params;
    
    const accountNumber = await BkashAccountNumber.findByPk(id);
    if (!accountNumber) {
      return res.status(404).json({ message: "Account number not found" });
    }

    // Remove primary status from other accounts of same type
    await BkashAccountNumber.update(
      { isPrimary: false },
      { 
        where: { 
          accountType: accountNumber.accountType, 
          isPrimary: true,
          id: { [Op.ne]: id }
        } 
      }
    );

    // Set this account as primary
    await accountNumber.update({ isPrimary: true });
    
    const updatedAccountNumber = await BkashAccountNumber.findByPk(id);
    res.json(updatedAccountNumber);
  } catch (error) {
    console.error('Error setting primary account:', error);
    res.status(500).json({ 
      message: "Error setting primary account",
      error: error.message 
    });
  }
};

// Get primary account number by type
exports.getPrimaryAccountByType = async (req, res) => {
  try {
    const { accountType } = req.params;
    
    const primaryAccount = await BkashAccountNumber.findOne({
      where: { 
        accountType: accountType,
        isPrimary: true,
        isActive: true
      }
    });

    if (!primaryAccount) {
      return res.status(404).json({ message: "No primary account found for this type" });
    }

    res.json(primaryAccount);
  } catch (error) {
    console.error('Error getting primary account:', error);
    res.status(500).json({ 
      message: "Error retrieving primary account",
      error: error.message 
    });
  }
};

// Toggle account status
exports.toggleAccountStatus = async (req, res) => {
  try {
    const { id } = req.params;
    
    const accountNumber = await BkashAccountNumber.findByPk(id);
    if (!accountNumber) {
      return res.status(404).json({ message: "Account number not found" });
    }

    await accountNumber.update({ isActive: !accountNumber.isActive });
    
    const updatedAccountNumber = await BkashAccountNumber.findByPk(id);
    res.json(updatedAccountNumber);
  } catch (error) {
    console.error('Error toggling account status:', error);
    res.status(500).json({ 
      message: "Error toggling account status",
      error: error.message 
    });
  }
};