const db = require('../models');
const BkashAccountNumber = db.bkashAccountNumbers;
const { Op } = require('sequelize');

// Create a new bKash account number
exports.create = async (req, res) => {
  try {
    const { 
      accountType, 
      accountNumber, 
      accountName, 
      accountLabel, 
      isActive = true, 
      isPrimary = false, 
      notes 
    } = req.body;

    // Validate required fields
    if (!accountType || !accountNumber || !accountName) {
      return res.status(400).send({
        message: "Account type, number, and name are required!"
      });
    }

    // Check if account number already exists
    const existingAccount = await BkashAccountNumber.findOne({
      where: { accountNumber }
    });

    if (existingAccount) {
      return res.status(400).send({
        message: "Account number already exists!"
      });
    }

    // If this is set as primary, unset other primary accounts of the same type
    if (isPrimary) {
      await BkashAccountNumber.update(
        { isPrimary: false },
        { where: { accountType, isPrimary: true } }
      );
    }

    // Create the account number
    const accountNumberRecord = await BkashAccountNumber.create({
      accountType,
      accountNumber,
      accountName,
      accountLabel,
      isActive,
      isPrimary,
      notes
    });

    res.send({
      message: "bKash account number created successfully!",
      data: accountNumberRecord
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the account number."
    });
  }
};

// Get all bKash account numbers
exports.findAll = async (req, res) => {
  try {
    const { accountType, isActive } = req.query;
    
    let whereCondition = {};
    
    if (accountType) {
      whereCondition.accountType = accountType;
    }
    
    if (isActive !== undefined) {
      whereCondition.isActive = isActive === 'true';
    }

    const accountNumbers = await BkashAccountNumber.findAll({
      where: whereCondition,
      order: [['isPrimary', 'DESC'], ['accountName', 'ASC']]
    });

    res.send(accountNumbers);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving account numbers."
    });
  }
};

// Get a single bKash account number by id
exports.findOne = async (req, res) => {
  try {
    const id = req.params.id;

    const accountNumber = await BkashAccountNumber.findByPk(id);

    if (!accountNumber) {
      return res.status(404).send({
        message: "Account number not found with id " + id
      });
    }

    res.send(accountNumber);
  } catch (err) {
    res.status(500).send({
      message: "Error retrieving account number with id=" + req.params.id
    });
  }
};

// Update a bKash account number by id
exports.update = async (req, res) => {
  try {
    const id = req.params.id;
    const { isPrimary, accountType } = req.body;

    // If this is being set as primary, unset other primary accounts of the same type
    if (isPrimary && accountType) {
      await BkashAccountNumber.update(
        { isPrimary: false },
        { where: { accountType, isPrimary: true, id: { [Op.ne]: id } } }
      );
    }

    const [num] = await BkashAccountNumber.update(req.body, {
      where: { id }
    });

    if (num == 1) {
      res.send({
        message: "Account number was updated successfully."
      });
    } else {
      res.send({
        message: `Cannot update account number with id=${id}. Maybe it was not found or req.body is empty!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: "Error updating account number with id=" + req.params.id
    });
  }
};

// Delete a bKash account number by id
exports.delete = async (req, res) => {
  try {
    const id = req.params.id;

    const num = await BkashAccountNumber.destroy({
      where: { id }
    });

    if (num == 1) {
      res.send({
        message: "Account number was deleted successfully!"
      });
    } else {
      res.send({
        message: `Cannot delete account number with id=${id}. Maybe it was not found!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: "Could not delete account number with id=" + req.params.id
    });
  }
};

// Set an account as primary
exports.setPrimary = async (req, res) => {
  try {
    const id = req.params.id;

    // Get the account to find its type
    const account = await BkashAccountNumber.findByPk(id);
    if (!account) {
      return res.status(404).send({
        message: "Account number not found with id " + id
      });
    }

    // Unset other primary accounts of the same type
    await BkashAccountNumber.update(
      { isPrimary: false },
      { where: { accountType: account.accountType, isPrimary: true } }
    );

    // Set this account as primary
    await BkashAccountNumber.update(
      { isPrimary: true },
      { where: { id } }
    );

    res.send({
      message: "Account set as primary successfully!"
    });
  } catch (err) {
    res.status(500).send({
      message: "Error setting account as primary with id=" + req.params.id
    });
  }
};