const db = require("../models");
const BkashMerchantAccount = db.bkashMerchantAccounts;
const BkashTransaction = db.bkashTransactions;
const BkashAccountNumber = db.bkashAccountNumbers;
const { Op } = require("sequelize");

// Get today's merchant account
exports.getTodayAccount = async (req, res) => {
  try {
    const today = new Date().toISOString().split('T')[0];
    
    let account = await BkashMerchantAccount.findOne({
      where: { date: today }
    });

    if (!account) {
      // Create today's account with opening balance from yesterday's closing
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      const yesterdayStr = yesterday.toISOString().split('T')[0];
      
      const yesterdayAccount = await BkashMerchantAccount.findOne({
        where: { date: yesterdayStr },
        order: [['date', 'DESC']]
      });

      const openingBalance = yesterdayAccount ? yesterdayAccount.closingBalance : 0;

      account = await BkashMerchantAccount.create({
        date: today,
        openingBalance: openingBalance || 0
      });
    }

    res.json(account);
  } catch (error) {
    console.error('Error getting today\'s merchant account:', error);
    res.status(500).json({ 
      message: "Error retrieving today's merchant account",
      error: error.message 
    });
  }
};

// Update merchant account
exports.updateAccount = async (req, res) => {
  try {
    const { date } = req.params;
    const updateData = req.body;

    const [updatedRowsCount] = await BkashMerchantAccount.update(updateData, {
      where: { date: date }
    });

    if (updatedRowsCount === 0) {
      return res.status(404).json({ message: "Merchant account not found for this date" });
    }

    const updatedAccount = await BkashMerchantAccount.findOne({
      where: { date: date }
    });

    res.json(updatedAccount);
  } catch (error) {
    console.error('Error updating merchant account:', error);
    res.status(500).json({ 
      message: "Error updating merchant account",
      error: error.message 
    });
  }
};

// Get merchant account by date
exports.getAccountByDate = async (req, res) => {
  try {
    const { date } = req.params;
    
    const account = await BkashMerchantAccount.findOne({
      where: { date: date }
    });

    if (!account) {
      return res.status(404).json({ message: "Merchant account not found for this date" });
    }

    res.json(account);
  } catch (error) {
    console.error('Error getting merchant account by date:', error);
    res.status(500).json({ 
      message: "Error retrieving merchant account",
      error: error.message 
    });
  }
};

// Get merchant transactions for a date
exports.getTransactionsByDate = async (req, res) => {
  try {
    const { date } = req.params;
    
    const transactions = await BkashTransaction.findAll({
      where: { 
        date: date,
        accountType: 'merchant'
      },
      include: [
        {
          model: BkashAccountNumber,
          as: 'bkashAccountNumber',
          required: false
        }
      ],
      order: [['time', 'DESC']]
    });

    res.json(transactions);
  } catch (error) {
    console.error('Error getting merchant transactions:', error);
    res.status(500).json({ 
      message: "Error retrieving merchant transactions",
      error: error.message 
    });
  }
};

// Add merchant transaction
exports.addTransaction = async (req, res) => {
  try {
    const { date } = req.params;
    const transactionData = {
      ...req.body,
      date: date,
      accountType: 'merchant'
    };

    // Get or create today's merchant account
    let merchantAccount = await BkashMerchantAccount.findOne({
      where: { date: date }
    });

    if (!merchantAccount) {
      merchantAccount = await BkashMerchantAccount.create({
        date: date,
        openingBalance: 0
      });
    }

    transactionData.merchantAccountId = merchantAccount.id;

    const transaction = await BkashTransaction.create(transactionData);

    // Update merchant account totals based on transaction type
    const updateData = {};
    switch (transaction.type) {
      case 'merchant_sale':
        updateData.totalSales = (parseFloat(merchantAccount.totalSales) || 0) + parseFloat(transaction.amount);
        break;
      case 'merchant_refund':
        updateData.totalRefunds = (parseFloat(merchantAccount.totalRefunds) || 0) + parseFloat(transaction.amount);
        break;
      case 'merchant_withdrawal':
        updateData.totalWithdrawals = (parseFloat(merchantAccount.totalWithdrawals) || 0) + parseFloat(transaction.amount);
        break;
      case 'merchant_fee':
        updateData.merchantFee = (parseFloat(merchantAccount.merchantFee) || 0) + parseFloat(transaction.amount);
        break;
    }

    if (Object.keys(updateData).length > 0) {
      await BkashMerchantAccount.update(updateData, {
        where: { id: merchantAccount.id }
      });
    }

    const createdTransaction = await BkashTransaction.findByPk(transaction.id, {
      include: [
        {
          model: BkashAccountNumber,
          as: 'bkashAccountNumber',
          required: false
        }
      ]
    });

    res.status(201).json(createdTransaction);
  } catch (error) {
    console.error('Error adding merchant transaction:', error);
    res.status(500).json({ 
      message: "Error adding merchant transaction",
      error: error.message 
    });
  }
};

// Delete merchant transaction
exports.deleteTransaction = async (req, res) => {
  try {
    const { transactionId } = req.params;
    
    const transaction = await BkashTransaction.findByPk(transactionId);
    if (!transaction) {
      return res.status(404).json({ message: "Transaction not found" });
    }

    // Get the merchant account to update totals
    const merchantAccount = await BkashMerchantAccount.findByPk(transaction.merchantAccountId);
    
    if (merchantAccount) {
      const updateData = {};
      switch (transaction.type) {
        case 'merchant_sale':
          updateData.totalSales = Math.max(0, (parseFloat(merchantAccount.totalSales) || 0) - parseFloat(transaction.amount));
          break;
        case 'merchant_refund':
          updateData.totalRefunds = Math.max(0, (parseFloat(merchantAccount.totalRefunds) || 0) - parseFloat(transaction.amount));
          break;
        case 'merchant_withdrawal':
          updateData.totalWithdrawals = Math.max(0, (parseFloat(merchantAccount.totalWithdrawals) || 0) - parseFloat(transaction.amount));
          break;
        case 'merchant_fee':
          updateData.merchantFee = Math.max(0, (parseFloat(merchantAccount.merchantFee) || 0) - parseFloat(transaction.amount));
          break;
      }

      if (Object.keys(updateData).length > 0) {
        await BkashMerchantAccount.update(updateData, {
          where: { id: merchantAccount.id }
        });
      }
    }

    await transaction.destroy();
    res.json({ message: "Transaction deleted successfully" });
  } catch (error) {
    console.error('Error deleting merchant transaction:', error);
    res.status(500).json({ 
      message: "Error deleting merchant transaction",
      error: error.message 
    });
  }
};

// Get merchant account history
exports.getAccountHistory = async (req, res) => {
  try {
    const { limit = 30 } = req.query;
    
    const accounts = await BkashMerchantAccount.findAll({
      order: [['date', 'DESC']],
      limit: parseInt(limit)
    });

    res.json(accounts);
  } catch (error) {
    console.error('Error getting merchant account history:', error);
    res.status(500).json({ 
      message: "Error retrieving merchant account history",
      error: error.message 
    });
  }
};