const db = require('../models');
const BkashAgentAccount = db.bkashAgentAccounts;
const BkashPersonalAccount = db.bkashPersonalAccounts;
const BkashTransaction = db.bkashTransactions;
const Sale = db.sales;
const { Op } = require('sequelize');

// Get daily report
exports.getDailyReport = async (req, res) => {
  try {
    const { date } = req.params;
    const reportDate = date || new Date().toISOString().split('T')[0];

    // Get agent account for the date
    const agentAccount = await BkashAgentAccount.findOne({
      where: { date: reportDate },
      include: [{
        model: BkashTransaction,
        required: false
      }]
    });

    // Get personal account for the date
    const personalAccount = await BkashPersonalAccount.findOne({
      where: { date: reportDate },
      include: [{
        model: BkashTransaction,
        required: false
      }]
    });

    // Get all transactions for the date
    const transactions = await BkashTransaction.findAll({
      where: { date: reportDate },
      order: [['createdAt', 'ASC']]
    });

    // Get sales data for the date
    const salesData = await Sale.findAll({
      where: {
        createdAt: {
          [Op.between]: [
            new Date(reportDate + 'T00:00:00.000Z'),
            new Date(reportDate + 'T23:59:59.999Z')
          ]
        }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('totalAmount')), 'totalSales'],
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'totalOrders']
      ]
    });

    const report = {
      date: reportDate,
      agentAccount: agentAccount || null,
      personalAccount: personalAccount || null,
      transactions,
      salesSummary: salesData[0] || { totalSales: 0, totalOrders: 0 },
      summary: {
        agentProfit: agentAccount ? agentAccount.dailyProfit : 0,
        personalNetChange: personalAccount ? personalAccount.netChange : 0,
        totalTransactions: transactions.length,
        agentCompleted: agentAccount ? agentAccount.isCompleted : false,
        personalCompleted: personalAccount ? personalAccount.isCompleted : false
      }
    };

    res.send(report);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while generating daily report."
    });
  }
};

// Get weekly report
exports.getWeeklyReport = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    let weekStart, weekEnd;
    if (startDate && endDate) {
      weekStart = startDate;
      weekEnd = endDate;
    } else {
      // Default to current week (Monday to Sunday)
      const today = new Date();
      const dayOfWeek = today.getDay();
      const diff = today.getDate() - dayOfWeek + (dayOfWeek === 0 ? -6 : 1);
      weekStart = new Date(today.setDate(diff)).toISOString().split('T')[0];
      weekEnd = new Date(today.setDate(diff + 6)).toISOString().split('T')[0];
    }

    // Get agent accounts for the week
    const agentAccounts = await BkashAgentAccount.findAll({
      where: {
        date: { [Op.between]: [weekStart, weekEnd] }
      },
      order: [['date', 'ASC']]
    });

    // Get personal accounts for the week
    const personalAccounts = await BkashPersonalAccount.findAll({
      where: {
        date: { [Op.between]: [weekStart, weekEnd] }
      },
      order: [['date', 'ASC']]
    });

    // Get transactions for the week
    const transactions = await BkashTransaction.findAll({
      where: {
        date: { [Op.between]: [weekStart, weekEnd] }
      },
      order: [['date', 'ASC'], ['createdAt', 'ASC']]
    });

    // Get sales data for the week
    const salesData = await Sale.findAll({
      where: {
        createdAt: {
          [Op.between]: [
            new Date(weekStart + 'T00:00:00.000Z'),
            new Date(weekEnd + 'T23:59:59.999Z')
          ]
        }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('totalAmount')), 'totalSales'],
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'totalOrders']
      ]
    });

    // Calculate totals
    const totalAgentProfit = agentAccounts.reduce((sum, acc) => sum + (acc.dailyProfit || 0), 0);
    const totalPersonalNetChange = personalAccounts.reduce((sum, acc) => sum + (acc.netChange || 0), 0);

    const report = {
      period: { startDate: weekStart, endDate: weekEnd },
      agentAccounts,
      personalAccounts,
      transactions,
      salesSummary: salesData[0] || { totalSales: 0, totalOrders: 0 },
      summary: {
        totalAgentProfit,
        totalPersonalNetChange,
        totalTransactions: transactions.length,
        completedDays: agentAccounts.filter(acc => acc.isCompleted).length,
        activeDays: agentAccounts.length
      }
    };

    res.send(report);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while generating weekly report."
    });
  }
};

// Get monthly report
exports.getMonthlyReport = async (req, res) => {
  try {
    const { month, year } = req.query;
    
    let monthStart, monthEnd;
    if (month && year) {
      monthStart = `${year}-${month.padStart(2, '0')}-01`;
      const lastDay = new Date(year, month, 0).getDate();
      monthEnd = `${year}-${month.padStart(2, '0')}-${lastDay}`;
    } else {
      // Default to current month
      const today = new Date();
      const currentYear = today.getFullYear();
      const currentMonth = today.getMonth() + 1;
      monthStart = `${currentYear}-${currentMonth.toString().padStart(2, '0')}-01`;
      const lastDay = new Date(currentYear, currentMonth, 0).getDate();
      monthEnd = `${currentYear}-${currentMonth.toString().padStart(2, '0')}-${lastDay}`;
    }

    // Get agent accounts for the month
    const agentAccounts = await BkashAgentAccount.findAll({
      where: {
        date: { [Op.between]: [monthStart, monthEnd] }
      },
      order: [['date', 'ASC']]
    });

    // Get personal accounts for the month
    const personalAccounts = await BkashPersonalAccount.findAll({
      where: {
        date: { [Op.between]: [monthStart, monthEnd] }
      },
      order: [['date', 'ASC']]
    });

    // Get transactions grouped by type
    const transactionSummary = await BkashTransaction.findAll({
      where: {
        date: { [Op.between]: [monthStart, monthEnd] }
      },
      attributes: [
        'type',
        'accountType',
        [db.sequelize.fn('SUM', db.sequelize.col('amount')), 'totalAmount'],
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'count']
      ],
      group: ['type', 'accountType']
    });

    // Get sales data for the month
    const salesData = await Sale.findAll({
      where: {
        createdAt: {
          [Op.between]: [
            new Date(monthStart + 'T00:00:00.000Z'),
            new Date(monthEnd + 'T23:59:59.999Z')
          ]
        }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('totalAmount')), 'totalSales'],
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'totalOrders'],
        [db.sequelize.fn('AVG', db.sequelize.col('totalAmount')), 'averageOrderValue']
      ]
    });

    // Calculate daily averages and totals
    const totalAgentProfit = agentAccounts.reduce((sum, acc) => sum + (acc.dailyProfit || 0), 0);
    const totalPersonalNetChange = personalAccounts.reduce((sum, acc) => sum + (acc.netChange || 0), 0);
    const activeDays = agentAccounts.length;
    const avgDailyProfit = activeDays > 0 ? totalAgentProfit / activeDays : 0;

    const report = {
      period: { startDate: monthStart, endDate: monthEnd },
      agentAccounts,
      personalAccounts,
      transactionSummary,
      salesSummary: salesData[0] || { totalSales: 0, totalOrders: 0, averageOrderValue: 0 },
      summary: {
        totalAgentProfit,
        totalPersonalNetChange,
        avgDailyProfit,
        activeDays,
        completedDays: agentAccounts.filter(acc => acc.isCompleted).length,
        totalTransactions: transactionSummary.reduce((sum, t) => sum + parseInt(t.dataValues.count), 0)
      }
    };

    res.send(report);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while generating monthly report."
    });
  }
};

// Get comparison report (Bkash vs Sales)
exports.getComparisonReport = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    if (!startDate || !endDate) {
      return res.status(400).send({
        message: "Start date and end date are required for comparison report."
      });
    }

    // Get Bkash summary
    const agentSummary = await BkashAgentAccount.findAll({
      where: {
        date: { [Op.between]: [startDate, endDate] }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('dailyProfit')), 'totalProfit'],
        [db.sequelize.fn('SUM', db.sequelize.col('srReceived')), 'totalSrReceived'],
        [db.sequelize.fn('SUM', db.sequelize.col('srGiven')), 'totalSrGiven'],
        [db.sequelize.fn('SUM', db.sequelize.col('bankDeposit')), 'totalBankDeposit'],
        [db.sequelize.fn('SUM', db.sequelize.col('bankWithdraw')), 'totalBankWithdraw']
      ]
    });

    const personalSummary = await BkashPersonalAccount.findAll({
      where: {
        date: { [Op.between]: [startDate, endDate] }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('netChange')), 'totalNetChange'],
        [db.sequelize.fn('SUM', db.sequelize.col('totalSent')), 'totalSent'],
        [db.sequelize.fn('SUM', db.sequelize.col('totalReceived')), 'totalReceived']
      ]
    });

    // Get sales summary
    const salesSummary = await Sale.findAll({
      where: {
        createdAt: {
          [Op.between]: [
            new Date(startDate + 'T00:00:00.000Z'),
            new Date(endDate + 'T23:59:59.999Z')
          ]
        }
      },
      attributes: [
        [db.sequelize.fn('SUM', db.sequelize.col('totalAmount')), 'totalSales'],
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'totalOrders'],
        [db.sequelize.fn('AVG', db.sequelize.col('totalAmount')), 'averageOrderValue']
      ]
    });

    const report = {
      period: { startDate, endDate },
      bkashSummary: {
        agent: agentSummary[0] || {},
        personal: personalSummary[0] || {}
      },
      salesSummary: salesSummary[0] || {},
      comparison: {
        bkashProfitVsSales: {
          agentProfit: agentSummary[0]?.dataValues?.totalProfit || 0,
          totalSales: salesSummary[0]?.dataValues?.totalSales || 0,
          profitPercentage: salesSummary[0]?.dataValues?.totalSales > 0 
            ? ((agentSummary[0]?.dataValues?.totalProfit || 0) / salesSummary[0].dataValues.totalSales * 100).toFixed(2)
            : 0
        }
      }
    };

    res.send(report);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while generating comparison report."
    });
  }
};