const db = require('../models');
const CashReconciliation = db.cashReconciliations;
const { Op } = require('sequelize');

// Get today's cash reconciliation
const getTodayReconciliation = async (req, res) => {
  try {
    const today = new Date().toISOString().split('T')[0];
    
    let reconciliation = await CashReconciliation.findOne({
      where: { date: today }
    });

    if (!reconciliation) {
      // Create a new record for today
      reconciliation = await CashReconciliation.create({
        date: today
      });
    }

    res.json(reconciliation);
  } catch (error) {
    console.error('Error fetching today reconciliation:', error);
    res.status(500).json({ error: 'Failed to fetch today reconciliation' });
  }
};

// Update cash reconciliation
const updateReconciliation = async (req, res) => {
  try {
    const { date } = req.params;
    const updateData = req.body;

    let reconciliation = await CashReconciliation.findOne({
      where: { date }
    });

    if (!reconciliation) {
      // Create new record
      reconciliation = await CashReconciliation.create({
        date,
        ...updateData
      });
    } else {
      // Update existing record
      await reconciliation.update(updateData);
    }

    res.json(reconciliation);
  } catch (error) {
    console.error('Error updating reconciliation:', error);
    res.status(500).json({ error: 'Failed to update reconciliation' });
  }
};

// Get reconciliation by date
const getReconciliationByDate = async (req, res) => {
  try {
    const { date } = req.params;
    
    const reconciliation = await CashReconciliation.findOne({
      where: { date }
    });

    if (!reconciliation) {
      return res.status(404).json({ error: 'Reconciliation not found for this date' });
    }

    res.json(reconciliation);
  } catch (error) {
    console.error('Error fetching reconciliation:', error);
    res.status(500).json({ error: 'Failed to fetch reconciliation' });
  }
};

// Get reconciliations for date range
const getReconciliationsByDateRange = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const whereClause = {};
    if (startDate && endDate) {
      whereClause.date = {
        [Op.between]: [startDate, endDate]
      };
    } else if (startDate) {
      whereClause.date = {
        [Op.gte]: startDate
      };
    } else if (endDate) {
      whereClause.date = {
        [Op.lte]: endDate
      };
    }

    const reconciliations = await CashReconciliation.findAll({
      where: whereClause,
      order: [['date', 'DESC']]
    });

    res.json(reconciliations);
  } catch (error) {
    console.error('Error fetching reconciliations:', error);
    res.status(500).json({ error: 'Failed to fetch reconciliations' });
  }
};

// Get weekly report
const getWeeklyReport = async (req, res) => {
  try {
    const { startDate } = req.query;
    const start = startDate || new Date(Date.now() - 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
    const end = new Date().toISOString().split('T')[0];

    const reconciliations = await CashReconciliation.findAll({
      where: {
        date: {
          [Op.between]: [start, end]
        }
      },
      order: [['date', 'ASC']]
    });

    // Calculate summary
    const summary = {
      totalDays: reconciliations.length,
      totalOpeningCash: reconciliations.reduce((sum, r) => sum + parseFloat(r.opening_total || 0), 0),
      totalClosingCash: reconciliations.reduce((sum, r) => sum + parseFloat(r.closing_total || 0), 0),
      totalVariance: reconciliations.reduce((sum, r) => sum + parseFloat(r.variance || 0), 0),
      averageVariance: reconciliations.length > 0 ? 
        reconciliations.reduce((sum, r) => sum + parseFloat(r.variance || 0), 0) / reconciliations.length : 0
    };

    res.json({
      reconciliations,
      summary,
      period: { start, end }
    });
  } catch (error) {
    console.error('Error fetching weekly report:', error);
    res.status(500).json({ error: 'Failed to fetch weekly report' });
  }
};

// Get monthly report
const getMonthlyReport = async (req, res) => {
  try {
    const { year, month } = req.query;
    const currentDate = new Date();
    const targetYear = year || currentDate.getFullYear();
    const targetMonth = month || (currentDate.getMonth() + 1);

    const startDate = `${targetYear}-${targetMonth.toString().padStart(2, '0')}-01`;
    const endDate = new Date(targetYear, targetMonth, 0).toISOString().split('T')[0];

    const reconciliations = await CashReconciliation.findAll({
      where: {
        date: {
          [Op.between]: [startDate, endDate]
        }
      },
      order: [['date', 'ASC']]
    });

    // Calculate summary
    const summary = {
      totalDays: reconciliations.length,
      totalOpeningCash: reconciliations.reduce((sum, r) => sum + parseFloat(r.opening_total || 0), 0),
      totalClosingCash: reconciliations.reduce((sum, r) => sum + parseFloat(r.closing_total || 0), 0),
      totalVariance: reconciliations.reduce((sum, r) => sum + parseFloat(r.variance || 0), 0),
      averageVariance: reconciliations.length > 0 ? 
        reconciliations.reduce((sum, r) => sum + parseFloat(r.variance || 0), 0) / reconciliations.length : 0
    };

    res.json({
      reconciliations,
      summary,
      period: { start: startDate, end: endDate, year: targetYear, month: targetMonth }
    });
  } catch (error) {
    console.error('Error fetching monthly report:', error);
    res.status(500).json({ error: 'Failed to fetch monthly report' });
  }
};

// Delete reconciliation
const deleteReconciliation = async (req, res) => {
  try {
    const { date } = req.params;
    
    const reconciliation = await CashReconciliation.findOne({
      where: { date }
    });

    if (!reconciliation) {
      return res.status(404).json({ error: 'Reconciliation not found' });
    }

    await reconciliation.destroy();
    res.json({ message: 'Reconciliation deleted successfully' });
  } catch (error) {
    console.error('Error deleting reconciliation:', error);
    res.status(500).json({ error: 'Failed to delete reconciliation' });
  }
};

module.exports = {
  getTodayReconciliation,
  updateReconciliation,
  getReconciliationByDate,
  getReconciliationsByDateRange,
  getWeeklyReport,
  getMonthlyReport,
  deleteReconciliation
};