const db = require('../models');
const MobileRechargeAccount = db.mobileRechargeAccounts;
const MobileRechargeTransaction = db.mobileRechargeTransactions;
const { Op } = require('sequelize');

// Get today's account for a specific operator
exports.getTodayAccount = async (req, res) => {
  try {
    const { operator } = req.params;
    const today = new Date().toISOString().split('T')[0];

    let account = await MobileRechargeAccount.findOne({
      where: { 
        date: today,
        operator: operator
      }
    });

    if (!account) {
      // Create today's account if it doesn't exist
      account = await MobileRechargeAccount.create({
        date: today,
        operator: operator,
        openingBalance: 0,
        totalRecharge: 0,
        totalCommission: 0
      });
    }

    res.send(account);
  } catch (error) {
    console.error('Error getting today\'s recharge account:', error);
    res.status(500).send({
      message: "Error retrieving today's recharge account",
      error: error.message
    });
  }
};

// Update or create recharge account
exports.updateAccount = async (req, res) => {
  try {
    const { operator } = req.params;
    const { 
      date, 
      openingBalance, 
      closingBalance, 
      totalRecharge, 
      totalCommission, 
      commissionRate,
      notes 
    } = req.body;

    const accountDate = date || new Date().toISOString().split('T')[0];

    let account = await MobileRechargeAccount.findOne({
      where: { 
        date: accountDate,
        operator: operator
      }
    });

    if (account) {
      const updatedOpeningBalance = openingBalance !== undefined ? parseFloat(openingBalance) : parseFloat(account.openingBalance) || 0;
      const updatedCommissionRate = commissionRate !== undefined ? parseFloat(commissionRate) : parseFloat(account.commissionRate) || 2.80;
      
      let updatedTotalRecharge = totalRecharge !== undefined ? parseFloat(totalRecharge) : parseFloat(account.totalRecharge) || 0;
      let updatedTotalCommission = totalCommission !== undefined ? parseFloat(totalCommission) : parseFloat(account.totalCommission) || 0;
      
      // Auto-calculate total recharge if closing balance is provided but total recharge is not provided in request
      if (closingBalance !== undefined && totalRecharge === undefined && updatedOpeningBalance > 0) {
        const calculatedRecharge = updatedOpeningBalance - parseFloat(closingBalance);
        if (calculatedRecharge > 0) {
          updatedTotalRecharge = calculatedRecharge;
          // Auto-calculate commission based on recharge amount and commission rate
          updatedTotalCommission = (updatedTotalRecharge * updatedCommissionRate) / 100;
        }
      }
      
      // Calculate automatic end-of-day balance if not manually provided
      let calculatedClosingBalance = closingBalance;
      if (closingBalance === undefined) {
        calculatedClosingBalance = updatedOpeningBalance - updatedTotalRecharge + updatedTotalCommission;
      } else {
        calculatedClosingBalance = parseFloat(closingBalance);
      }

      // Calculate net change
      const calculatedNetChange = calculatedClosingBalance - updatedOpeningBalance;

      await account.update({
        openingBalance: updatedOpeningBalance,
        closingBalance: calculatedClosingBalance,
        totalRecharge: updatedTotalRecharge,
        totalCommission: updatedTotalCommission,
        commissionRate: updatedCommissionRate,
        notes: notes !== undefined ? notes : account.notes,
        netChange: calculatedNetChange
      });
    } else {
      const newOpeningBalance = parseFloat(openingBalance) || 0;
      const newCommissionRate = parseFloat(commissionRate) || 2.80;
      
      let newTotalRecharge = parseFloat(totalRecharge) || 0;
      let newTotalCommission = parseFloat(totalCommission) || 0;
      
      // Auto-calculate total recharge if closing balance is provided but total recharge is not
      if (closingBalance !== undefined && totalRecharge === undefined && newOpeningBalance > 0) {
        const calculatedRecharge = newOpeningBalance - parseFloat(closingBalance);
        if (calculatedRecharge > 0) {
          newTotalRecharge = calculatedRecharge;
          // Auto-calculate commission based on recharge amount and commission rate
          newTotalCommission = (newTotalRecharge * newCommissionRate) / 100;
        }
      }
      
      // Calculate automatic end-of-day balance if not manually provided
      let calculatedClosingBalance = closingBalance;
      if (closingBalance === undefined) {
        calculatedClosingBalance = newOpeningBalance - newTotalRecharge + newTotalCommission;
      } else {
        calculatedClosingBalance = parseFloat(closingBalance);
      }

      // Calculate net change
      const calculatedNetChange = calculatedClosingBalance - newOpeningBalance;

      account = await MobileRechargeAccount.create({
        date: accountDate,
        operator: operator,
        openingBalance: newOpeningBalance,
        closingBalance: calculatedClosingBalance,
        totalRecharge: newTotalRecharge,
        totalCommission: newTotalCommission,
        commissionRate: newCommissionRate,
        notes: notes,
        netChange: calculatedNetChange
      });
    }

    res.send(account);
  } catch (error) {
    console.error('Error updating recharge account:', error);
    res.status(500).send({
      message: "Error updating recharge account",
      error: error.message
    });
  }
};

// Get account by date and operator
exports.getAccountByDate = async (req, res) => {
  try {
    const { operator, date } = req.params;

    const account = await MobileRechargeAccount.findOne({
      where: { 
        date: date,
        operator: operator
      }
    });

    if (!account) {
      return res.status(404).send({
        message: `No recharge account found for ${operator} on ${date}`
      });
    }

    res.send(account);
  } catch (error) {
    console.error('Error getting recharge account by date:', error);
    res.status(500).send({
      message: "Error retrieving recharge account",
      error: error.message
    });
  }
};

// Get transactions for a specific operator and date
exports.getTransactions = async (req, res) => {
  try {
    const { operator } = req.params;
    const { date } = req.query;
    const transactionDate = date || new Date().toISOString().split('T')[0];

    const transactions = await MobileRechargeTransaction.findAll({
      where: {
        operator: operator,
        date: transactionDate
      },
      order: [['time', 'DESC']]
    });

    res.send(transactions);
  } catch (error) {
    console.error('Error getting recharge transactions:', error);
    res.status(500).send({
      message: "Error retrieving recharge transactions",
      error: error.message
    });
  }
};

// Add a new recharge transaction
exports.addTransaction = async (req, res) => {
  try {
    const { operator } = req.params;
    const {
      customerNumber,
      rechargeAmount,
      commission,
      srPay,
      minutesCard,
      mbCard,
      simQuantity,
      transactionType,
      reference,
      notes,
      date,
      time
    } = req.body;

    // Validate required fields
    if (!customerNumber || !rechargeAmount) {
      return res.status(400).send({
        message: "Customer number and recharge amount are required!"
      });
    }

    const transactionDate = date || new Date().toISOString().split('T')[0];
    const transactionTime = time || new Date().toTimeString().split(' ')[0];

    // Create the transaction
    const transaction = await MobileRechargeTransaction.create({
      date: transactionDate,
      time: transactionTime,
      operator: operator,
      customerNumber: customerNumber,
      rechargeAmount: rechargeAmount,
      commission: commission || 0,
      srPay: srPay || 0,
      minutesCard: minutesCard || 0,
      mbCard: mbCard || 0,
      simQuantity: simQuantity || 0,
      transactionType: transactionType || 'recharge',
      reference: reference,
      notes: notes
    });

    // Update the account totals
    let account = await MobileRechargeAccount.findOne({
      where: { 
        date: transactionDate,
        operator: operator
      }
    });

    if (!account) {
      account = await MobileRechargeAccount.create({
        date: transactionDate,
        operator: operator,
        openingBalance: 0,
        totalRecharge: rechargeAmount,
        totalCommission: commission || 0,
        totalSrPay: srPay || 0,
        totalMinutesCard: minutesCard || 0,
        totalMbCard: mbCard || 0,
        totalSimQuantity: simQuantity || 0
      });
    } else {
      const newTotalRecharge = parseFloat(account.totalRecharge) + parseFloat(rechargeAmount);
      const newTotalCommission = parseFloat(account.totalCommission) + parseFloat(commission || 0);
      const openingBalance = parseFloat(account.openingBalance) || 0;
      
      // Automatically calculate new closing balance
      const calculatedClosingBalance = openingBalance + newTotalRecharge + newTotalCommission;
      const calculatedNetChange = calculatedClosingBalance - openingBalance;

      await account.update({
        totalRecharge: newTotalRecharge,
        totalCommission: newTotalCommission,
        totalSrPay: parseFloat(account.totalSrPay) + parseFloat(srPay || 0),
        totalMinutesCard: parseFloat(account.totalMinutesCard) + parseFloat(minutesCard || 0),
        totalMbCard: parseFloat(account.totalMbCard) + parseFloat(mbCard || 0),
        totalSimQuantity: parseInt(account.totalSimQuantity) + parseInt(simQuantity || 0),
        closingBalance: calculatedClosingBalance,
        netChange: calculatedNetChange
      });
    }

    res.send(transaction);
  } catch (error) {
    console.error('Error adding recharge transaction:', error);
    res.status(500).send({
      message: "Error adding recharge transaction",
      error: error.message
    });
  }
};

// Delete a recharge transaction
exports.deleteTransaction = async (req, res) => {
  try {
    const { id } = req.params;

    const transaction = await MobileRechargeTransaction.findByPk(id);
    if (!transaction) {
      return res.status(404).send({
        message: "Transaction not found!"
      });
    }

    // Update account totals before deleting
    const account = await MobileRechargeAccount.findOne({
      where: { 
        date: transaction.date,
        operator: transaction.operator
      }
    });

    if (account) {
      const newTotalRecharge = parseFloat(account.totalRecharge) - parseFloat(transaction.rechargeAmount);
      const newTotalCommission = parseFloat(account.totalCommission) - parseFloat(transaction.commission);
      const openingBalance = parseFloat(account.openingBalance) || 0;
      
      // Automatically recalculate closing balance after transaction deletion
      const calculatedClosingBalance = openingBalance + newTotalRecharge + newTotalCommission;
      const calculatedNetChange = calculatedClosingBalance - openingBalance;

      await account.update({
        totalRecharge: newTotalRecharge,
        totalCommission: newTotalCommission,
        totalSrPay: parseFloat(account.totalSrPay) - parseFloat(transaction.srPay || 0),
        totalMinutesCard: parseFloat(account.totalMinutesCard) - parseFloat(transaction.minutesCard || 0),
        totalMbCard: parseFloat(account.totalMbCard) - parseFloat(transaction.mbCard || 0),
        totalSimQuantity: parseInt(account.totalSimQuantity) - parseInt(transaction.simQuantity || 0),
        closingBalance: calculatedClosingBalance,
        netChange: calculatedNetChange
      });
    }

    await transaction.destroy();

    res.send({
      message: "Transaction deleted successfully!"
    });
  } catch (error) {
    console.error('Error deleting recharge transaction:', error);
    res.status(500).send({
      message: "Error deleting recharge transaction",
      error: error.message
    });
  }
};

// Get account history for an operator
exports.getAccountHistory = async (req, res) => {
  try {
    const { operator } = req.params;
    const { startDate, endDate, limit = 30 } = req.query;

    const whereClause = { operator: operator };
    
    if (startDate && endDate) {
      whereClause.date = {
        [Op.between]: [startDate, endDate]
      };
    }

    const accounts = await MobileRechargeAccount.findAll({
      where: whereClause,
      order: [['date', 'DESC']],
      limit: parseInt(limit)
    });

    res.send(accounts);
  } catch (error) {
    console.error('Error getting recharge account history:', error);
    res.status(500).send({
      message: "Error retrieving recharge account history",
      error: error.message
    });
  }
};

// Get summary for all operators
exports.getSummary = async (req, res) => {
  try {
    const { date } = req.query;
    const summaryDate = date || new Date().toISOString().split('T')[0];

    const operators = ['grameenphone', 'banglalink', 'robi'];
    const summary = {};

    for (const operator of operators) {
      const account = await MobileRechargeAccount.findOne({
        where: { 
          date: summaryDate,
          operator: operator
        }
      });

      const transactionCount = await MobileRechargeTransaction.count({
        where: {
          operator: operator,
          date: summaryDate
        }
      });

      summary[operator] = {
        openingBalance: account ? account.openingBalance : 0,
        closingBalance: account ? account.closingBalance : null,
        totalRecharge: account ? account.totalRecharge : 0,
        totalCommission: account ? account.totalCommission : 0,
        netChange: account ? account.netChange : null,
        transactionCount: transactionCount,
        isCompleted: account ? account.isCompleted : false
      };
    }

    res.send(summary);
  } catch (error) {
    console.error('Error getting recharge summary:', error);
    res.status(500).send({
      message: "Error retrieving recharge summary",
      error: error.message
    });
  }
};

// Open day with starting balance
exports.openDay = async (req, res) => {
  try {
    const { operator } = req.params;
    const { openingBalance, date, notes } = req.body;
    const accountDate = date || new Date().toISOString().split('T')[0];

    // Check if day is already opened
    let account = await MobileRechargeAccount.findOne({
      where: { 
        date: accountDate,
        operator: operator
      }
    });

    if (account && account.openingBalance !== null) {
      return res.status(400).send({
        message: `Day already opened for ${operator} on ${accountDate}`,
        account: account
      });
    }

    if (account) {
      // Update existing account with opening balance
      await account.update({
        openingBalance: parseFloat(openingBalance) || 0,
        notes: notes || account.notes,
        isCompleted: false
      });
    } else {
      // Create new account for the day
      account = await MobileRechargeAccount.create({
        date: accountDate,
        operator: operator,
        openingBalance: parseFloat(openingBalance) || 0,
        totalRecharge: 0,
        totalCommission: 0,
        totalSrPay: 0,
        totalMinutesCard: 0,
        totalMbCard: 0,
        totalSimQuantity: 0,
        notes: notes,
        isCompleted: false
      });
    }

    res.send({
      message: `Day opened successfully for ${operator}`,
      account: account
    });
  } catch (error) {
    console.error('Error opening day:', error);
    res.status(500).send({
      message: "Error opening day",
      error: error.message
    });
  }
};

// Close day with ending balance
exports.closeDay = async (req, res) => {
  try {
    const { operator } = req.params;
    const { closingBalance, date, notes } = req.body;
    const accountDate = date || new Date().toISOString().split('T')[0];

    const account = await MobileRechargeAccount.findOne({
      where: { 
        date: accountDate,
        operator: operator
      }
    });

    if (!account) {
      return res.status(404).send({
        message: `No account found for ${operator} on ${accountDate}. Please open the day first.`
      });
    }

    if (account.isCompleted) {
      return res.status(400).send({
        message: `Day already closed for ${operator} on ${accountDate}`,
        account: account
      });
    }

    // Calculate net change
    const netChange = parseFloat(closingBalance) - parseFloat(account.openingBalance);

    await account.update({
      closingBalance: parseFloat(closingBalance),
      netChange: netChange,
      notes: notes || account.notes,
      isCompleted: true
    });

    res.send({
      message: `Day closed successfully for ${operator}`,
      account: account,
      netChange: netChange
    });
  } catch (error) {
    console.error('Error closing day:', error);
    res.status(500).send({
      message: "Error closing day",
      error: error.message
    });
  }
};

// Get daily summary with balance calculations
exports.getDailySummary = async (req, res) => {
  try {
    const { operator } = req.params;
    const { date } = req.query;
    const summaryDate = date || new Date().toISOString().split('T')[0];

    const account = await MobileRechargeAccount.findOne({
      where: { 
        date: summaryDate,
        operator: operator
      }
    });

    const transactionCount = await MobileRechargeTransaction.count({
      where: {
        operator: operator,
        date: summaryDate
      }
    });

    const recentTransactions = await MobileRechargeTransaction.findAll({
      where: {
        operator: operator,
        date: summaryDate
      },
      order: [['time', 'DESC']],
      limit: 5
    });

    const summary = {
      date: summaryDate,
      operator: operator,
      isOpened: account ? account.openingBalance !== null : false,
      isClosed: account ? account.isCompleted : false,
      openingBalance: account ? account.openingBalance : null,
      closingBalance: account ? account.closingBalance : null,
      totalRecharge: account ? account.totalRecharge : 0,
      totalCommission: account ? account.totalCommission : 0,
      totalSrPay: account ? account.totalSrPay : 0,
      totalMinutesCard: account ? account.totalMinutesCard : 0,
      totalMbCard: account ? account.totalMbCard : 0,
      totalSimQuantity: account ? account.totalSimQuantity : 0,
      netChange: account ? account.netChange : null,
      transactionCount: transactionCount,
      recentTransactions: recentTransactions,
      notes: account ? account.notes : null
    };

    res.send(summary);
  } catch (error) {
    console.error('Error getting daily summary:', error);
    res.status(500).send({
      message: "Error retrieving daily summary",
      error: error.message
    });
  }
};

// Get consolidated daily mobile recharge summary for dashboard
exports.getDashboardSummary = async (req, res) => {
  try {
    const { date } = req.query;
    const summaryDate = date || new Date().toISOString().split('T')[0];

    const operators = ['grameenphone', 'banglalink', 'robi'];
    let totalOpeningBalance = 0;
    let totalClosingBalance = 0;
    let totalRecharge = 0;
    let totalCommission = 0;
    let totalTransactions = 0;
    let totalNetChange = 0;
    let operatorsOpened = 0;
    let operatorsClosed = 0;

    const operatorDetails = {};

    for (const operator of operators) {
      const account = await MobileRechargeAccount.findOne({
        where: { 
          date: summaryDate,
          operator: operator
        }
      });

      const transactionCount = await MobileRechargeTransaction.count({
        where: {
          operator: operator,
          date: summaryDate
        }
      });

      const operatorData = {
        openingBalance: account ? parseFloat(account.openingBalance) || 0 : 0,
        closingBalance: account ? parseFloat(account.closingBalance) || 0 : 0,
        totalRecharge: account ? parseFloat(account.totalRecharge) || 0 : 0,
        totalCommission: account ? parseFloat(account.totalCommission) || 0 : 0,
        netChange: account ? parseFloat(account.netChange) || 0 : 0,
        transactionCount: transactionCount,
        isOpened: account ? account.openingBalance !== null : false,
        isClosed: account ? account.isCompleted : false
      };

      operatorDetails[operator] = operatorData;

      // Aggregate totals
      totalOpeningBalance += operatorData.openingBalance;
      totalClosingBalance += operatorData.closingBalance;
      totalRecharge += operatorData.totalRecharge;
      totalCommission += operatorData.totalCommission;
      totalNetChange += operatorData.netChange;
      totalTransactions += transactionCount;

      if (operatorData.isOpened) operatorsOpened++;
      if (operatorData.isClosed) operatorsClosed++;
    }

    // Calculate end-of-day balance automatically
    const calculatedEndBalance = totalOpeningBalance + totalRecharge + totalCommission;

    const dashboardSummary = {
      date: summaryDate,
      totalOpeningBalance: totalOpeningBalance,
      totalClosingBalance: totalClosingBalance,
      calculatedEndBalance: calculatedEndBalance,
      totalRecharge: totalRecharge,
      totalCommission: totalCommission,
      totalNetChange: totalNetChange,
      totalTransactions: totalTransactions,
      operatorsOpened: operatorsOpened,
      operatorsClosed: operatorsClosed,
      totalOperators: operators.length,
      operatorDetails: operatorDetails,
      status: {
        allOpened: operatorsOpened === operators.length,
        allClosed: operatorsClosed === operators.length,
        partiallyOpened: operatorsOpened > 0 && operatorsOpened < operators.length
      }
    };

    res.send(dashboardSummary);
  } catch (error) {
    console.error('Error getting mobile recharge dashboard summary:', error);
    res.status(500).send({
      message: "Error retrieving mobile recharge dashboard summary",
      error: error.message
    });
  }
};