const db = require("../models");
const Note = db.notes;
const Op = db.Sequelize.Op;

// Create a new note
exports.create = (req, res) => {
  // Validate request
  if (!req.body.content) {
    res.status(400).send({
      message: "Content cannot be empty!"
    });
    return;
  }

  // Create a note
  const note = {
    title: req.body.title || "Daily Note",
    content: req.body.content,
    date: req.body.date || new Date().toISOString().split('T')[0],
    type: req.body.type || 'note',
    isCompleted: req.body.isCompleted || false,
    priority: req.body.priority || 'medium',
    userId: req.userId
  };

  // Save note in the database
  Note.create(note)
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while creating the note."
      });
    });
};

// Retrieve all notes for a user
exports.findAll = (req, res) => {
  const date = req.query.date;
  const type = req.query.type;
  
  let condition = { userId: req.userId };
  
  if (date) {
    condition.date = date;
  }
  
  if (type) {
    condition.type = type;
  }

  Note.findAll({ 
    where: condition,
    order: [['createdAt', 'DESC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving notes."
      });
    });
};

// Find today's notes for a user
exports.findToday = (req, res) => {
  const today = new Date().toISOString().split('T')[0];
  
  Note.findAll({
    where: {
      userId: req.userId,
      date: today
    },
    order: [['createdAt', 'DESC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving today's notes."
      });
    });
};

// Find a single note with an id
exports.findOne = (req, res) => {
  const id = req.params.id;

  Note.findOne({
    where: {
      id: id,
      userId: req.userId
    }
  })
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        res.status(404).send({
          message: `Cannot find note with id=${id}.`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving note with id=" + id
      });
    });
};

// Update a note by the id in the request
exports.update = (req, res) => {
  const id = req.params.id;

  Note.update(req.body, {
    where: { 
      id: id,
      userId: req.userId
    }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Note was updated successfully."
        });
      } else {
        res.send({
          message: `Cannot update note with id=${id}. Maybe note was not found or req.body is empty!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error updating note with id=" + id
      });
    });
};

// Delete a note with the specified id in the request
exports.delete = (req, res) => {
  const id = req.params.id;

  Note.destroy({
    where: { 
      id: id,
      userId: req.userId
    }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Note was deleted successfully!"
        });
      } else {
        res.send({
          message: `Cannot delete note with id=${id}. Maybe note was not found!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Could not delete note with id=" + id
      });
    });
};

// Toggle completion status for todo items
exports.toggleComplete = (req, res) => {
  const id = req.params.id;

  Note.findOne({
    where: {
      id: id,
      userId: req.userId,
      type: 'todo'
    }
  })
    .then(note => {
      if (note) {
        note.update({ isCompleted: !note.isCompleted })
          .then(() => {
            res.send({
              message: "Todo status updated successfully.",
              isCompleted: note.isCompleted
            });
          })
          .catch(err => {
            res.status(500).send({
              message: "Error updating todo status with id=" + id
            });
          });
      } else {
        res.status(404).send({
          message: `Cannot find todo with id=${id}.`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving todo with id=" + id
      });
    });
};