module.exports = (sequelize, Sequelize) => {
  const BkashAgentAccount = sequelize.define("bkashAgentAccount", {
    id: {
      type: Sequelize.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    date: {
      type: Sequelize.DATEONLY,
      allowNull: false,
      unique: true,
      defaultValue: Sequelize.literal('CURRENT_DATE')
    },
    openingBalance: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Cash/float available at start of day"
    },
    closingBalance: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Cash/float remaining at end of day"
    },
    srReceived: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money received from Bkash SR"
    },
    srGiven: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money given to Bkash SR"
    },
    bankDeposit: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money deposited to bank"
    },
    bankWithdraw: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money withdrawn from bank"
    },
    b2bSend: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money sent via B2B transactions"
    },
    b2bReceived: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Money received via B2B transactions"
    },
    amountReceived: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Total amount received from customers/transactions"
    },
    totalTransaction: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Auto-calculated: Total transaction amount (Opening - Closing)"
    },
    commissionRate: {
      type: Sequelize.DECIMAL(5, 2),
      allowNull: false,
      defaultValue: 0.40,
      comment: "Commission rate percentage (default 0.4%)"
    },
    totalCommission: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Auto-calculated: Total commission earned (totalTransaction * commissionRate / 100)"
    },
    bkashAccountNumberId: {
      type: Sequelize.INTEGER,
      allowNull: true,
      references: {
        model: 'bkash_account_numbers',
        key: 'id'
      },
      comment: "Reference to the specific bKash account number used"
    },
    dailyProfit: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Auto-calculated: Closing - Opening + Received - Given"
    },
    notes: {
      type: Sequelize.TEXT,
      allowNull: true,
      comment: "Additional notes for the day"
    },
    isCompleted: {
      type: Sequelize.BOOLEAN,
      allowNull: false,
      defaultValue: false,
      comment: "Whether the day is completed (closing balance entered)"
    }
  }, {
    indexes: [
      {
        unique: true,
        fields: ['date']
      }
    ],
    hooks: {
      beforeSave: (instance) => {
        // Auto-calculate total transaction and commission if closing balance is set
        if (instance.closingBalance !== null && instance.openingBalance !== null) {
          // Calculate total transaction amount (Opening - Closing)
          instance.totalTransaction = parseFloat(instance.openingBalance) - parseFloat(instance.closingBalance);
          
          // Calculate total commission (totalTransaction * commissionRate / 100)
          if (instance.totalTransaction > 0) {
            const commissionRate = parseFloat(instance.commissionRate || 0.40);
            instance.totalCommission = (instance.totalTransaction * commissionRate) / 100;
          } else {
            instance.totalCommission = 0;
          }
        }

        // Auto-calculate daily profit if closing balance is set
        if (instance.closingBalance !== null) {
          const netSrTransaction = parseFloat(instance.srReceived || 0) - parseFloat(instance.srGiven || 0);
          const netBankTransaction = parseFloat(instance.bankWithdraw || 0) - parseFloat(instance.bankDeposit || 0);
          const netB2bTransaction = parseFloat(instance.b2bReceived || 0) - parseFloat(instance.b2bSend || 0);
          const totalReceived = parseFloat(instance.amountReceived || 0);
          instance.dailyProfit = parseFloat(instance.closingBalance) - parseFloat(instance.openingBalance) + netSrTransaction + netBankTransaction + netB2bTransaction + totalReceived;
          instance.isCompleted = true;
        }
      }
    }
  });

  return BkashAgentAccount;
};