module.exports = (sequelize, Sequelize) => {
  const BkashMerchantAccount = sequelize.define("bkashMerchantAccount", {
    id: {
      type: Sequelize.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    date: {
      type: Sequelize.DATEONLY,
      allowNull: false,
      unique: true,
      defaultValue: Sequelize.literal('CURRENT_DATE')
    },
    openingBalance: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Merchant Bkash balance at start of day"
    },
    closingBalance: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Merchant Bkash balance at end of day"
    },
    totalSales: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Total sales amount received during the day"
    },
    totalRefunds: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Total refunds given during the day"
    },
    totalWithdrawals: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Total amount withdrawn to bank/cash"
    },
    merchantFee: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      comment: "Merchant fees charged by Bkash"
    },
    netRevenue: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Auto-calculated: Sales - Refunds - Fees"
    },
    netChange: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: true,
      defaultValue: null,
      comment: "Auto-calculated: Closing - Opening"
    },
    notes: {
      type: Sequelize.TEXT,
      allowNull: true,
      comment: "Additional notes for the day"
    },
    isCompleted: {
      type: Sequelize.BOOLEAN,
      allowNull: false,
      defaultValue: false,
      comment: "Whether the day's accounting is completed"
    }
  }, {
    tableName: 'bkash_merchant_accounts',
    timestamps: true,
    hooks: {
      beforeSave: (account, options) => {
        // Auto-calculate net revenue
        if (account.totalSales !== null && account.totalRefunds !== null && account.merchantFee !== null) {
          account.netRevenue = parseFloat(account.totalSales) - parseFloat(account.totalRefunds) - parseFloat(account.merchantFee);
        }
        
        // Auto-calculate net change
        if (account.openingBalance !== null && account.closingBalance !== null) {
          account.netChange = parseFloat(account.closingBalance) - parseFloat(account.openingBalance);
        }
      }
    }
  });

  return BkashMerchantAccount;
};