module.exports = (sequelize, Sequelize) => {
  const DSOAllocation = sequelize.define("dsoAllocation", {
    id: {
      type: Sequelize.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    deliveryId: {
      type: Sequelize.INTEGER,
      allowNull: true,
      comment: "Reference to DSO delivery (optional for standalone allocations)"
    },
    date: {
      type: Sequelize.DATEONLY,
      allowNull: false,
      defaultValue: Sequelize.literal('CURRENT_DATE'),
      comment: "Allocation date"
    },
    cardsAllocated: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of cards allocated for sale"
    },
    cardTypesAllocated: {
      type: Sequelize.JSON,
      allowNull: true,
      defaultValue: [],
      comment: "Breakdown of allocated cards by type: [{type: string, quantity: number}]"
    },
    simsAllocated: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of SIMs allocated for sale"
    },
    kitsAllocated: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of kits allocated for sale"
    },
    cardsSold: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of allocated cards sold"
    },
    cardTypesSold: {
      type: Sequelize.JSON,
      allowNull: true,
      defaultValue: [],
      comment: "Breakdown of sold cards by type: [{type: string, quantity: number}]"
    },
    simsSold: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of allocated SIMs sold"
    },
    kitsSold: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of allocated kits sold"
    },
    allocatedBy: {
      type: Sequelize.STRING(100),
      allowNull: false,
      comment: "User who made the allocation"
    },
    status: {
      type: Sequelize.ENUM('active', 'completed', 'cancelled'),
      allowNull: false,
      defaultValue: 'active',
      comment: "Allocation status"
    },
    notes: {
      type: Sequelize.TEXT,
      allowNull: true,
      comment: "Additional notes about the allocation"
    }
  }, {
    timestamps: true,
    createdAt: 'allocatedAt',
    updatedAt: 'updatedAt',
    hooks: {
      beforeValidate: (allocation, options) => {
        // Validate that at least one item type is allocated
        if (allocation.cardsAllocated === 0 && allocation.simsAllocated === 0 && allocation.kitsAllocated === 0) {
          throw new Error('At least one of Cards, SIMs, or Kits must be allocated (quantity > 0)');
        }
        
        // Validate card types allocation total matches cardsAllocated
        if (allocation.cardTypesAllocated && allocation.cardTypesAllocated.length > 0) {
          const totalFromTypes = allocation.cardTypesAllocated.reduce((sum, cardType) => sum + (cardType.quantity || 0), 0);
          if (totalFromTypes !== allocation.cardsAllocated) {
            throw new Error('Sum of allocated card type quantities must equal total cards allocated');
          }
        }
        
        // Validate sold quantities don't exceed allocated quantities
        if (allocation.cardsSold > allocation.cardsAllocated) {
          throw new Error('Cards sold cannot exceed cards allocated');
        }
        if (allocation.simsSold > allocation.simsAllocated) {
          throw new Error('SIMs sold cannot exceed SIMs allocated');
        }
        if (allocation.kitsSold > allocation.kitsAllocated) {
          throw new Error('Kits sold cannot exceed kits allocated');
        }
      }
    }
  });

  return DSOAllocation;
};