const express = require('express');
const cors = require('cors');
const db = require('./models');

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Import routes
const productRoutes = require('./routes/product.routes');
const categoryRoutes = require('./routes/category.routes');
const inventoryRoutes = require('./routes/inventory.routes');
const salesRoutes = require('./routes/sales.routes');
const authRoutes = require('./routes/auth.routes');
const agentRoutes = require('./routes/agent.routes');
const reportsRoutes = require('./routes/reports.routes');
const customerRoutes = require('./routes/customer.routes');
const expenseRoutes = require('./routes/expense.routes');

// Import Bkash routes
require('./routes/bkashAgent.routes')(app);
require('./routes/bkashPersonal.routes')(app);
require('./routes/bkashReports.routes')(app);
require('./routes/bkashAccountNumbers.routes')(app);
const bkashMerchantRoutes = require('./routes/bkashMerchantAccount.routes');
const bkashAccountNumberRoutes = require('./routes/bkashAccountNumber.routes');
const bkashTransactionsRoutes = require('./routes/bkashTransactions.routes');

// Import Mobile Recharge routes
require('./routes/mobileRecharge.routes')(app);

// Import Notes routes
require('./routes/note.routes')(app);

// Import Cash Reconciliation routes
const cashReconciliationRoutes = require('./routes/cashReconciliation.routes');

// Import SMS routes
const smsRoutes = require('./routes/sms.routes');

// Import DSO routes
const dsoRoutes = require('./routes/dso.routes');

// Use routes
app.use('/api/products', productRoutes);
app.use('/api/categories', categoryRoutes);
app.use('/api/inventory', inventoryRoutes);
app.use('/api/sales', salesRoutes);
app.use('/api/auth', authRoutes);
app.use('/api/agents', agentRoutes);
app.use('/api/reports', reportsRoutes);
app.use('/api/customers', customerRoutes);
app.use('/api/expenses', expenseRoutes);
app.use('/api/bkash-merchant', bkashMerchantRoutes);
app.use('/api/bkash-account-numbers', bkashAccountNumberRoutes);
app.use('/api/bkash-transactions', bkashTransactionsRoutes);
app.use('/api/cash-reconciliation', cashReconciliationRoutes);
app.use('/api/sms', smsRoutes);
app.use('/api/dso', dsoRoutes);

// Root route
app.get('/', (req, res) => {
  res.json({ 
    message: 'Welcome to POS System API',
    version: '1.0.0',
    endpoints: [
      '/api/products',
      '/api/categories',
      '/api/inventory',
      '/api/sales',
      '/api/auth',
      '/api/agents',
      '/api/reports',
      '/api/customers',
      '/api/bkash-agent',
      '/api/bkash-personal',
      '/api/bkash-merchant',
      '/api/bkash-account-numbers',
      '/api/bkash-transactions',
      '/api/bkash-reports',
      '/api/mobile-recharge',
      '/api/notes',
      '/api/cash-reconciliation',
      '/api/sms',
      '/api/dso'
    ]
  });
});

// Sync database
db.sequelize.sync()
  .then(() => {
    console.log('Database synced');
    // Create initial admin user if not exists
    const { users } = db;
    const bcrypt = require('bcryptjs');
    users.findOrCreate({
      where: { username: 'admin' },
      defaults: {
        username: 'admin',
        password: bcrypt.hashSync('admin123', 8),
        email: 'admin@example.com',
        fullName: 'System Administrator',
        role: 'admin',
        status: 'active'
      }
    }).then(([user, created]) => {
      if (created) {
        console.log('Admin user created successfully');
      } else {
        console.log('Admin user already exists');
      }
    });
  })
  .catch((err) => {
    console.error('Failed to sync database:', err);
  });

// Start server
app.listen(PORT, () => {
  console.log(`Server is running on port ${PORT}`);
  console.log(`API is available at http://localhost:${PORT}`);
});