import React, { useState, useEffect } from 'react';
import { Card, Badge, Spinner, Alert } from 'react-bootstrap';
import { FaSms, FaCheckCircle, FaExclamationCircle, FaEye } from 'react-icons/fa';
import { useNavigate } from 'react-router-dom';
import bkashSmsService from '../services/bkashSmsService';
import './BkashSmsWidget.css';

const BkashSmsWidget = () => {
  const [smsData, setSmsData] = useState({
    totalTransactions: 0,
    todayTransactions: 0,
    serverStatus: 'disconnected',
    lastUpdate: null,
    recentTransactions: []
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const navigate = useNavigate();

  useEffect(() => {
    fetchSmsData();
    const interval = setInterval(fetchSmsData, 30000); // Update every 30 seconds
    return () => clearInterval(interval);
  }, []);

  const fetchSmsData = async () => {
    try {
      setLoading(true);
      
      // Check server status
      const statusResult = await bkashSmsService.getConnectionStatus();
      const serverStatus = statusResult.success ? 'connected' : 'disconnected';
      
      if (statusResult.success) {
        // Fetch transactions
        const transactionsResult = await bkashSmsService.getTransactions();
        
        if (transactionsResult.success) {
          const transactions = transactionsResult.data;
          const today = new Date().toDateString();
          const todayTransactions = transactions.filter(t => 
            new Date(t.timestamp).toDateString() === today
          );

          setSmsData({
            totalTransactions: transactions.length,
            todayTransactions: todayTransactions.length,
            serverStatus: serverStatus,
            lastUpdate: new Date(),
            recentTransactions: transactions.slice(0, 3) // Last 3 transactions
          });
        } else {
          setSmsData(prev => ({
            ...prev,
            serverStatus: serverStatus,
            lastUpdate: new Date()
          }));
        }
      } else {
        setSmsData(prev => ({
          ...prev,
          serverStatus: serverStatus,
          lastUpdate: new Date()
        }));
      }
      
      setError('');
    } catch (error) {
      console.error('Error fetching SMS data:', error);
      setError('Failed to fetch SMS data');
      setSmsData(prev => ({
        ...prev,
        serverStatus: 'disconnected',
        lastUpdate: new Date()
      }));
    } finally {
      setLoading(false);
    }
  };

  const getStatusBadge = () => {
    if (smsData.serverStatus === 'connected') {
      return <Badge bg="success"><FaCheckCircle className="me-1" />Connected</Badge>;
    } else {
      return <Badge bg="danger"><FaExclamationCircle className="me-1" />Disconnected</Badge>;
    }
  };

  const handleViewAll = () => {
    navigate('/bkash/sms-transactions');
  };

  if (loading && !smsData.lastUpdate) {
    return (
      <Card className="bkash-sms-widget h-100">
        <Card.Body className="text-center">
          <Spinner animation="border" size="sm" />
          <p className="mt-2 mb-0">Loading SMS data...</p>
        </Card.Body>
      </Card>
    );
  }

  return (
    <Card className="bkash-sms-widget h-100 border-primary">
      <Card.Header className="bg-primary text-white d-flex justify-content-between align-items-center">
        <div>
          <FaSms className="me-2" />
          bKash SMS Integration
        </div>
        {getStatusBadge()}
      </Card.Header>
      <Card.Body>
        {error && (
          <Alert variant="warning" className="py-2 mb-3">
            <small>{error}</small>
          </Alert>
        )}
        
        <div className="row text-center mb-3">
          <div className="col-6">
            <h4 className="text-primary mb-1">{smsData.todayTransactions}</h4>
            <small className="text-muted">Today</small>
          </div>
          <div className="col-6">
            <h4 className="text-info mb-1">{smsData.totalTransactions}</h4>
            <small className="text-muted">Total</small>
          </div>
        </div>

        {smsData.recentTransactions.length > 0 && (
          <div className="recent-transactions">
            <h6 className="mb-2">Recent Transactions</h6>
            {smsData.recentTransactions.map((transaction, index) => (
              <div key={index} className="transaction-item mb-2 p-2 bg-light rounded">
                <div className="d-flex justify-content-between align-items-center">
                  <div>
                    <strong className="text-success">
                      {bkashSmsService.formatCurrency(transaction.amount)}
                    </strong>
                    <br />
                    <small className="text-muted">
                      {transaction.type} • {transaction.sender || 'N/A'}
                    </small>
                  </div>
                  <small className="text-muted">
                    {bkashSmsService.formatDate(transaction.timestamp).split(',')[1]}
                  </small>
                </div>
              </div>
            ))}
          </div>
        )}

        <div className="text-center mt-3">
          <button 
            className="btn btn-outline-primary btn-sm"
            onClick={handleViewAll}
          >
            <FaEye className="me-1" />
            View All Transactions
          </button>
        </div>

        {smsData.lastUpdate && (
          <div className="text-center mt-2">
            <small className="text-muted">
              Last updated: {smsData.lastUpdate.toLocaleTimeString()}
            </small>
          </div>
        )}
      </Card.Body>
    </Card>
  );
};

export default BkashSmsWidget;