import React, { useState, useEffect } from 'react';
import { Card, Table, Button, Badge, Modal, Form, Row, Col, Alert, Spinner } from 'react-bootstrap';
import { FaPlus, FaEdit, FaTrash, FaMoneyBillWave } from 'react-icons/fa';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../../contexts/AuthContext';
import axios from 'axios';
import './AgentList.css';

const AgentList = () => {
  const navigate = useNavigate();
  const { currentUser } = useAuth();
  
  // State for agents data
  const [agents, setAgents] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
  
  // State for modals
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [currentAgent, setCurrentAgent] = useState(null);
  const [submitting, setSubmitting] = useState(false);
  
  const [newAgent, setNewAgent] = useState({
    name: '',
    company: '',
    phone: '',
    email: '',
    address: '',
    status: 'active'
  });
  const [payment, setPayment] = useState({
    amount: '',
    type: 'payment',
    method: 'cash',
    reference: '',
    notes: ''
  });

  // Fetch agents from API
  useEffect(() => {
    if (!currentUser) {
      navigate('/login');
      return;
    }
    fetchAgents();
  }, [currentUser, navigate]);

  const fetchAgents = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      if (!token) {
        setError('Please log in to access agent management.');
        return;
      }
      
      const response = await axios.get('http://localhost:5000/api/agents', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAgents(response.data);
      setError('');
    } catch (error) {
      console.error('Error fetching agents:', error);
      
      if (error.response?.status === 401 || error.response?.status === 403) {
        setError('Authentication failed. Please log in again.');
        // Clear invalid token
        localStorage.removeItem('token');
        delete axios.defaults.headers.common['Authorization'];
      } else if (error.response?.status === 404) {
        setError('Agent service not found. Please contact support.');
      } else {
        setError(`Failed to fetch agents: ${error.response?.data?.message || error.message}`);
      }
    } finally {
      setLoading(false);
    }
  };

  // Show success message
  const showSuccess = (message) => {
    setSuccess(message);
    setTimeout(() => setSuccess(''), 3000);
  };

  // Show error message
  const showError = (message) => {
    setError(message);
    setTimeout(() => setError(''), 5000);
  };
  
  // Handle form input changes
  const handleAgentChange = (e) => {
    const { name, value } = e.target;
    setNewAgent({
      ...newAgent,
      [name]: value
    });
  };
  
  const handlePaymentChange = (e) => {
    const { name, value } = e.target;
    setPayment({
      ...payment,
      [name]: value
    });
  };
  
  // Handle add agent
  const handleAddAgent = async () => {
    try {
      setSubmitting(true);
      setError('');
      
      // Validate required fields
      if (!newAgent.name.trim() || !newAgent.phone.trim()) {
        showError('Name and phone number are required.');
        return;
      }

      const token = localStorage.getItem('token');
      const response = await axios.post('http://localhost:5000/api/agents', newAgent, {
        headers: { Authorization: `Bearer ${token}` }
      });

      // Add the new agent to the list
      setAgents([...agents, response.data]);
      setShowAddModal(false);
      setNewAgent({
        name: '',
        company: '',
        phone: '',
        email: '',
        address: '',
        status: 'active'
      });
      showSuccess('Agent added successfully!');
    } catch (error) {
      console.error('Error adding agent:', error);
      
      if (error.response?.status === 401 || error.response?.status === 403) {
        showError('Authentication failed. Please log in again.');
        localStorage.removeItem('token');
        delete axios.defaults.headers.common['Authorization'];
      } else if (error.response?.data?.message) {
        showError(error.response.data.message);
      } else {
        showError('Failed to add agent. Please try again.');
      }
    } finally {
      setSubmitting(false);
    }
  };
  
  // Handle edit agent
  const handleEditAgent = async () => {
    try {
      setSubmitting(true);
      setError('');
      
      // Validate required fields
      if (!newAgent.name.trim() || !newAgent.phone.trim()) {
        showError('Name and phone number are required.');
        return;
      }

      const token = localStorage.getItem('token');
      await axios.put(`http://localhost:5000/api/agents/${currentAgent.id}`, newAgent, {
        headers: { Authorization: `Bearer ${token}` }
      });

      // Update the agent in the list
      const updatedAgents = agents.map(agent => 
        agent.id === currentAgent.id ? { ...agent, ...newAgent } : agent
      );
      setAgents(updatedAgents);
      setShowEditModal(false);
      showSuccess('Agent updated successfully!');
    } catch (error) {
      console.error('Error updating agent:', error);
      if (error.response?.data?.message) {
        showError(error.response.data.message);
      } else {
        showError('Failed to update agent. Please try again.');
      }
    } finally {
      setSubmitting(false);
    }
  };
  
  // Handle delete agent
  const handleDeleteAgent = async () => {
    try {
      setSubmitting(true);
      setError('');

      const token = localStorage.getItem('token');
      await axios.delete(`http://localhost:5000/api/agents/${currentAgent.id}`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      // Remove the agent from the list
      const updatedAgents = agents.filter(agent => agent.id !== currentAgent.id);
      setAgents(updatedAgents);
      setShowDeleteModal(false);
      showSuccess('Agent deleted successfully!');
    } catch (error) {
      console.error('Error deleting agent:', error);
      if (error.response?.data?.message) {
        showError(error.response.data.message);
      } else {
        showError('Failed to delete agent. Please try again.');
      }
    } finally {
      setSubmitting(false);
    }
  };
  
  // Handle add payment
  const handleAddPayment = async () => {
    try {
      setSubmitting(true);
      setError('');
      
      // Validate required fields
      if (!payment.amount || parseFloat(payment.amount) <= 0) {
        showError('Please enter a valid amount.');
        return;
      }

      const token = localStorage.getItem('token');
      await axios.post(`http://localhost:5000/api/agents/${currentAgent.id}/payments`, payment, {
        headers: { Authorization: `Bearer ${token}` }
      });

      // Refresh agents to get updated balance
      await fetchAgents();
      setShowPaymentModal(false);
      setPayment({
        amount: '',
        type: 'payment',
        method: 'cash',
        reference: '',
        notes: ''
      });
      showSuccess('Payment added successfully!');
    } catch (error) {
      console.error('Error adding payment:', error);
      if (error.response?.data?.message) {
        showError(error.response.data.message);
      } else {
        showError('Failed to add payment. Please try again.');
      }
    } finally {
      setSubmitting(false);
    }
  };

  // Open edit modal
  const openEditModal = (agent) => {
    setCurrentAgent(agent);
    setNewAgent({
      name: agent.name,
      company: agent.company,
      phone: agent.phone,
      email: agent.email,
      address: agent.address || '',
      status: agent.status
    });
    setShowEditModal(true);
  };
  
  // Open delete modal
  const openDeleteModal = (agent) => {
    setCurrentAgent(agent);
    setShowDeleteModal(true);
  };
  
  // Open payment modal
  const openPaymentModal = (agent) => {
    setCurrentAgent(agent);
    setShowPaymentModal(true);
  };
  
  return (
    <div className="agent-list-container">
      <Card>
        <Card.Header className="d-flex justify-content-between align-items-center">
          <h4 className="mb-0">Agents & Suppliers</h4>
          <Button variant="primary" onClick={() => setShowAddModal(true)}>
            <FaPlus className="me-1" /> Add Agent
          </Button>
        </Card.Header>
        <Card.Body>
          {/* Success Alert */}
          {success && (
            <Alert variant="success" dismissible onClose={() => setSuccess('')}>
              {success}
            </Alert>
          )}
          
          {/* Error Alert */}
          {error && (
            <Alert variant="danger" dismissible onClose={() => setError('')}>
              {error}
            </Alert>
          )}
          
          {/* Loading Spinner */}
          {loading ? (
            <div className="text-center py-4">
              <Spinner animation="border" role="status">
                <span className="visually-hidden">Loading...</span>
              </Spinner>
              <div className="mt-2">Loading agents...</div>
            </div>
          ) : (
            <div className="table-responsive">
              <Table striped bordered hover>
              <thead>
                <tr>
                  <th>Name</th>
                  <th>Company</th>
                  <th>Contact</th>
                  <th>Balance</th>
                  <th>Status</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
                {agents.length === 0 ? (
                  <tr>
                    <td colSpan="6" className="text-center py-4">
                      <div className="text-muted">
                        <FaPlus size={48} className="mb-3 opacity-50" />
                        <div>No agents found</div>
                        <div className="small">Click "Add Agent" to create your first agent</div>
                      </div>
                    </td>
                  </tr>
                ) : (
                  agents.map((agent) => (
                    <tr key={agent.id}>
                      <td>{agent.name}</td>
                      <td>{agent.company}</td>
                      <td>
                        <div>{agent.phone}</div>
                        <div className="text-muted small">{agent.email}</div>
                      </td>
                      <td className={agent.balance < 0 ? 'text-danger' : agent.balance > 0 ? 'text-success' : ''}>
                        ৳{agent.balance?.toFixed(2) || '0.00'}
                      </td>
                      <td>
                        <Badge bg={agent.status === 'active' ? 'success' : 'secondary'}>
                          {agent.status}
                        </Badge>
                      </td>
                      <td>
                        <div className="d-flex gap-2">
                          <Button variant="outline-primary" size="sm" onClick={() => openPaymentModal(agent)}>
                            <FaMoneyBillWave />
                          </Button>
                          <Button variant="outline-secondary" size="sm" onClick={() => openEditModal(agent)}>
                            <FaEdit />
                          </Button>
                          <Button variant="outline-danger" size="sm" onClick={() => openDeleteModal(agent)}>
                            <FaTrash />
                          </Button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </Table>
          </div>
          )}
         </Card.Body>
       </Card>
      
      {/* Add Agent Modal */}
      <Modal show={showAddModal} onHide={() => setShowAddModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Add New Agent</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form>
            <Row>
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Name</Form.Label>
                <Form.Control
                  type="text"
                  name="name"
                  value={newAgent.name}
                  onChange={handleAgentChange}
                  placeholder="Enter agent name"
                  required
                />
              </Form.Group>
              
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Company</Form.Label>
                <Form.Control
                  type="text"
                  name="company"
                  value={newAgent.company}
                  onChange={handleAgentChange}
                  placeholder="Enter company name"
                />
              </Form.Group>
            </Row>
            
            <Row>
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Phone</Form.Label>
                <Form.Control
                  type="text"
                  name="phone"
                  value={newAgent.phone}
                  onChange={handleAgentChange}
                  placeholder="Enter phone number"
                  required
                />
              </Form.Group>
              
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Email</Form.Label>
                <Form.Control
                  type="email"
                  name="email"
                  value={newAgent.email}
                  onChange={handleAgentChange}
                  placeholder="Enter email address"
                />
              </Form.Group>
            </Row>
            
            <Form.Group className="mb-3">
              <Form.Label>Address</Form.Label>
              <Form.Control
                as="textarea"
                name="address"
                value={newAgent.address}
                onChange={handleAgentChange}
                placeholder="Enter address"
                rows={2}
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Status</Form.Label>
              <Form.Select
                name="status"
                value={newAgent.status}
                onChange={handleAgentChange}
              >
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </Form.Select>
            </Form.Group>
          </Form>
        </Modal.Body>
        <Modal.Footer>
           <Button variant="secondary" onClick={() => setShowAddModal(false)} disabled={submitting}>
             Cancel
           </Button>
           <Button variant="primary" onClick={handleAddAgent} disabled={submitting}>
             {submitting ? (
               <>
                 <Spinner animation="border" size="sm" className="me-2" />
                 Adding...
               </>
             ) : (
               'Add Agent'
             )}
           </Button>
         </Modal.Footer>
      </Modal>
      
      {/* Edit Agent Modal */}
      <Modal show={showEditModal} onHide={() => setShowEditModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Edit Agent</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form>
            <Row>
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Name</Form.Label>
                <Form.Control
                  type="text"
                  name="name"
                  value={newAgent.name}
                  onChange={handleAgentChange}
                  placeholder="Enter agent name"
                  required
                />
              </Form.Group>
              
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Company</Form.Label>
                <Form.Control
                  type="text"
                  name="company"
                  value={newAgent.company}
                  onChange={handleAgentChange}
                  placeholder="Enter company name"
                />
              </Form.Group>
            </Row>
            
            <Row>
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Phone</Form.Label>
                <Form.Control
                  type="text"
                  name="phone"
                  value={newAgent.phone}
                  onChange={handleAgentChange}
                  placeholder="Enter phone number"
                  required
                />
              </Form.Group>
              
              <Form.Group as={Col} md={6} className="mb-3">
                <Form.Label>Email</Form.Label>
                <Form.Control
                  type="email"
                  name="email"
                  value={newAgent.email}
                  onChange={handleAgentChange}
                  placeholder="Enter email address"
                />
              </Form.Group>
            </Row>
            
            <Form.Group className="mb-3">
              <Form.Label>Address</Form.Label>
              <Form.Control
                as="textarea"
                name="address"
                value={newAgent.address}
                onChange={handleAgentChange}
                placeholder="Enter address"
                rows={2}
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Status</Form.Label>
              <Form.Select
                name="status"
                value={newAgent.status}
                onChange={handleAgentChange}
              >
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </Form.Select>
            </Form.Group>
          </Form>
        </Modal.Body>
        <Modal.Footer>
           <Button variant="secondary" onClick={() => setShowEditModal(false)} disabled={submitting}>
             Cancel
           </Button>
           <Button variant="primary" onClick={handleEditAgent} disabled={submitting}>
             {submitting ? (
               <>
                 <Spinner animation="border" size="sm" className="me-2" />
                 Saving...
               </>
             ) : (
               'Save Changes'
             )}
           </Button>
         </Modal.Footer>
      </Modal>
      
      {/* Delete Confirmation Modal */}
      <Modal show={showDeleteModal} onHide={() => setShowDeleteModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Confirm Delete</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          Are you sure you want to delete agent "{currentAgent?.name}"? This action cannot be undone.
        </Modal.Body>
        <Modal.Footer>
           <Button variant="secondary" onClick={() => setShowDeleteModal(false)} disabled={submitting}>
             Cancel
           </Button>
           <Button variant="danger" onClick={handleDeleteAgent} disabled={submitting}>
             {submitting ? (
               <>
                 <Spinner animation="border" size="sm" className="me-2" />
                 Deleting...
               </>
             ) : (
               'Delete'
             )}
           </Button>
         </Modal.Footer>
      </Modal>
      
      {/* Payment Modal */}
      <Modal show={showPaymentModal} onHide={() => setShowPaymentModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Add Payment</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <div className="mb-3">
            <strong>Agent:</strong> {currentAgent?.name}
            <div className="text-muted small">{currentAgent?.company}</div>
            <div className="mt-2">
              <strong>Current Balance:</strong> <span className={currentAgent?.balance < 0 ? 'text-danger' : currentAgent?.balance > 0 ? 'text-success' : ''}>
                ৳{currentAgent?.balance.toFixed(2)}
              </span>
            </div>
          </div>
          
          <Form>
            <Form.Group className="mb-3">
              <Form.Label>Transaction Type</Form.Label>
              <Form.Select
                name="type"
                value={payment.type}
                onChange={handlePaymentChange}
              >
                <option value="payment">Payment (To Agent)</option>
                <option value="receipt">Receipt (From Agent)</option>
              </Form.Select>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Amount</Form.Label>
              <Form.Control
                type="number"
                name="amount"
                value={payment.amount}
                onChange={handlePaymentChange}
                placeholder="Enter amount"
                min="0"
                step="0.01"
                required
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Payment Method</Form.Label>
              <Form.Select
                name="method"
                value={payment.method}
                onChange={handlePaymentChange}
              >
                <option value="cash">Cash</option>
                <option value="bank">Bank Transfer</option>
                <option value="mobile">Mobile Banking</option>
              </Form.Select>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Reference</Form.Label>
              <Form.Control
                type="text"
                name="reference"
                value={payment.reference}
                onChange={handlePaymentChange}
                placeholder="Enter reference number/details"
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Notes</Form.Label>
              <Form.Control
                as="textarea"
                name="notes"
                value={payment.notes}
                onChange={handlePaymentChange}
                placeholder="Enter notes"
                rows={2}
              />
            </Form.Group>
          </Form>
        </Modal.Body>
        <Modal.Footer>
           <Button variant="secondary" onClick={() => setShowPaymentModal(false)} disabled={submitting}>
             Cancel
           </Button>
           <Button variant="primary" onClick={handleAddPayment} disabled={submitting}>
             {submitting ? (
               <>
                 <Spinner animation="border" size="sm" className="me-2" />
                 Saving...
               </>
             ) : (
               'Save Transaction'
             )}
           </Button>
         </Modal.Footer>
      </Modal>
    </div>
  );
};

export default AgentList;