import React, { useState, useEffect } from 'react';
import { Row, Col, Card, Form, Button, Table, Badge, Spinner, Modal, Tab, Tabs, Alert } from 'react-bootstrap';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';

const BkashAccountNumbers = ({ showAlert: propShowAlert }) => {
  const { currentUser } = useAuth();
  const [alert, setAlert] = useState({ show: false, message: '', variant: 'success' });

  const showAlert = propShowAlert || ((message, variant = 'success') => {
    setAlert({ show: true, message, variant });
    setTimeout(() => setAlert({ show: false, message: '', variant: 'success' }), 5000);
  });
  const [accountNumbers, setAccountNumbers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('all');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [editingAccount, setEditingAccount] = useState(null);
  const [formData, setFormData] = useState({
    accountType: 'agent',
    accountNumber: '',
    accountName: '',
    accountLabel: '',
    isPrimary: false,
    notes: ''
  });

  useEffect(() => {
    // Only fetch data if user is authenticated
    if (currentUser) {
      fetchAccountNumbers();
    }
  }, [currentUser]);

  const fetchAccountNumbers = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-account-numbers', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAccountNumbers(response.data);
    } catch (error) {
      console.error('Error fetching account numbers:', error);
      showAlert('Error fetching account numbers', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleInputChange = (e) => {
    const { name, value, type, checked } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.post('http://localhost:5000/api/bkash-account-numbers', formData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Account number added successfully!', 'success');
      setShowAddModal(false);
      setFormData({
        accountType: 'agent',
        accountNumber: '',
        accountName: '',
        accountLabel: '',
        isPrimary: false,
        notes: ''
      });
      fetchAccountNumbers();
    } catch (error) {
      console.error('Error adding account number:', error);
      showAlert(error.response?.data?.message || 'Error adding account number', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleSetPrimary = async (id) => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.patch(`http://localhost:5000/api/bkash-account-numbers/${id}/primary`, {}, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Primary account updated successfully!', 'success');
      fetchAccountNumbers();
    } catch (error) {
      console.error('Error setting primary account:', error);
      showAlert('Error setting primary account', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleToggleStatus = async (id) => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.patch(`http://localhost:5000/api/bkash-account-numbers/${id}/toggle-status`, {}, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Account status updated successfully!', 'success');
      fetchAccountNumbers();
    } catch (error) {
      console.error('Error toggling account status:', error);
      showAlert('Error updating account status', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleEdit = (account) => {
    setEditingAccount(account);
    setFormData({
      accountType: account.accountType,
      accountNumber: account.accountNumber,
      accountName: account.accountName,
      accountLabel: account.accountLabel || '',
      isPrimary: account.isPrimary,
      notes: account.notes || ''
    });
    setShowEditModal(true);
  };

  const handleUpdate = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.put(`http://localhost:5000/api/bkash-account-numbers/${editingAccount.id}`, formData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Account number updated successfully!', 'success');
      setShowEditModal(false);
      setEditingAccount(null);
      setFormData({
        accountType: 'agent',
        accountNumber: '',
        accountName: '',
        accountLabel: '',
        isPrimary: false,
        notes: ''
      });
      fetchAccountNumbers();
    } catch (error) {
      console.error('Error updating account number:', error);
      showAlert(error.response?.data?.message || 'Error updating account number', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleDelete = async (id) => {
    if (!window.confirm('Are you sure you want to delete this account number?')) {
      return;
    }

    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.delete(`http://localhost:5000/api/bkash-account-numbers/${id}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Account number deleted successfully!', 'success');
      fetchAccountNumbers();
    } catch (error) {
      console.error('Error deleting account number:', error);
      showAlert(error.response?.data?.message || 'Error deleting account number', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const getAccountTypeLabel = (type) => {
    switch (type) {
      case 'agent': return 'Agent';
      case 'personal': return 'Personal';
      case 'merchant': return 'Merchant';
      default: return type;
    }
  };

  const getAccountTypeBadge = (type) => {
    switch (type) {
      case 'agent': return 'primary';
      case 'personal': return 'success';
      case 'merchant': return 'info';
      default: return 'secondary';
    }
  };

  const filterAccountsByType = (type) => {
    if (type === 'all') return accountNumbers;
    return accountNumbers.filter(account => account.accountType === type);
  };

  const getAccountStats = () => {
    const stats = {
      total: accountNumbers.length,
      agent: accountNumbers.filter(a => a.accountType === 'agent').length,
      personal: accountNumbers.filter(a => a.accountType === 'personal').length,
      merchant: accountNumbers.filter(a => a.accountType === 'merchant').length,
      active: accountNumbers.filter(a => a.isActive).length,
      inactive: accountNumbers.filter(a => !a.isActive).length
    };
    return stats;
  };

  const stats = getAccountStats();

  return (
    <div className="bkash-account-numbers">
      {/* Alert */}
      {alert.show && (
        <Alert variant={alert.variant} dismissible onClose={() => setAlert({ show: false, message: '', variant: 'success' })}>
          {alert.message}
        </Alert>
      )}
      
      {/* Header */}
      <div className="page-header">
        <div className="d-flex justify-content-between align-items-center">
          <div>
            <h4 className="page-title">
              <i className="fas fa-credit-card me-2"></i>
              Bkash Account Numbers
            </h4>
            <p className="page-subtitle">Manage multiple account numbers for each account type</p>
          </div>
          <div className="header-actions">
            <Button 
              className="btn-bkash"
              onClick={() => setShowAddModal(true)}
            >
              <i className="fas fa-plus me-2"></i>
              Add Account Number
            </Button>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <Row className="mb-4">
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-primary">{stats.total}</h3>
              <p className="mb-0">Total Accounts</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-primary">{stats.agent}</h3>
              <p className="mb-0">Agent</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-success">{stats.personal}</h3>
              <p className="mb-0">Personal</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-info">{stats.merchant}</h3>
              <p className="mb-0">Merchant</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-success">{stats.active}</h3>
              <p className="mb-0">Active</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={2}>
          <Card className="stats-card">
            <Card.Body className="text-center">
              <h3 className="text-danger">{stats.inactive}</h3>
              <p className="mb-0">Inactive</p>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Account Numbers Tabs */}
      <Tabs activeKey={activeTab} onSelect={(k) => setActiveTab(k)} className="mb-3">
        <Tab eventKey="all" title={`All (${stats.total})`}>
          <AccountNumbersTable 
            accounts={filterAccountsByType('all')}
            loading={loading}
            onSetPrimary={handleSetPrimary}
            onToggleStatus={handleToggleStatus}
            onEdit={handleEdit}
            onDelete={handleDelete}
            getAccountTypeLabel={getAccountTypeLabel}
            getAccountTypeBadge={getAccountTypeBadge}
          />
        </Tab>
        <Tab eventKey="agent" title={`Agent (${stats.agent})`}>
          <AccountNumbersTable 
            accounts={filterAccountsByType('agent')}
            loading={loading}
            onSetPrimary={handleSetPrimary}
            onToggleStatus={handleToggleStatus}
            onEdit={handleEdit}
            onDelete={handleDelete}
            getAccountTypeLabel={getAccountTypeLabel}
            getAccountTypeBadge={getAccountTypeBadge}
          />
        </Tab>
        <Tab eventKey="personal" title={`Personal (${stats.personal})`}>
          <AccountNumbersTable 
            accounts={filterAccountsByType('personal')}
            loading={loading}
            onSetPrimary={handleSetPrimary}
            onToggleStatus={handleToggleStatus}
            onEdit={handleEdit}
            onDelete={handleDelete}
            getAccountTypeLabel={getAccountTypeLabel}
            getAccountTypeBadge={getAccountTypeBadge}
          />
        </Tab>
        <Tab eventKey="merchant" title={`Merchant (${stats.merchant})`}>
          <AccountNumbersTable 
            accounts={filterAccountsByType('merchant')}
            loading={loading}
            onSetPrimary={handleSetPrimary}
            onToggleStatus={handleToggleStatus}
            onEdit={handleEdit}
            onDelete={handleDelete}
            getAccountTypeLabel={getAccountTypeLabel}
            getAccountTypeBadge={getAccountTypeBadge}
          />
        </Tab>
      </Tabs>

      {/* Add Account Number Modal */}
      <Modal show={showAddModal} onHide={() => setShowAddModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>Add New Account Number</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form onSubmit={handleSubmit}>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Type</Form.Label>
                  <Form.Select
                    name="accountType"
                    value={formData.accountType}
                    onChange={handleInputChange}
                    required
                  >
                    <option value="agent">Agent</option>
                    <option value="personal">Personal</option>
                    <option value="merchant">Merchant</option>
                  </Form.Select>
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountNumber"
                    value={formData.accountNumber}
                    onChange={handleInputChange}
                    placeholder="Enter account number"
                    required
                  />
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col md={8}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Name</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountName"
                    value={formData.accountName}
                    onChange={handleInputChange}
                    placeholder="Enter account name (optional)"
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Check
                    type="checkbox"
                    name="isPrimary"
                    checked={formData.isPrimary}
                    onChange={handleInputChange}
                    label="Set as Primary"
                    className="mt-4"
                  />
                </Form.Group>
              </Col>
            </Row>

            <div className="d-flex gap-2">
              <Button 
                type="submit" 
                className="btn-bkash"
                disabled={loading}
              >
                {loading ? <Spinner animation="border" size="sm" /> : 'Add Account Number'}
              </Button>
              <Button 
                variant="outline-secondary"
                onClick={() => setShowAddModal(false)}
              >
                Cancel
              </Button>
            </div>
          </Form>
        </Modal.Body>
      </Modal>

      {/* Edit Account Number Modal */}
      <Modal show={showEditModal} onHide={() => setShowEditModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>Edit Account Number</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form onSubmit={handleUpdate}>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Type</Form.Label>
                  <Form.Select
                    name="accountType"
                    value={formData.accountType}
                    onChange={handleInputChange}
                    required
                  >
                    <option value="agent">Agent</option>
                    <option value="personal">Personal</option>
                    <option value="merchant">Merchant</option>
                  </Form.Select>
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountNumber"
                    value={formData.accountNumber}
                    onChange={handleInputChange}
                    placeholder="Enter account number"
                    required
                  />
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Name</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountName"
                    value={formData.accountName}
                    onChange={handleInputChange}
                    placeholder="Enter account name"
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Account Label</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountLabel"
                    value={formData.accountLabel}
                    onChange={handleInputChange}
                    placeholder="Enter account label (optional)"
                  />
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col md={8}>
                <Form.Group className="mb-3">
                  <Form.Label>Notes</Form.Label>
                  <Form.Control
                    as="textarea"
                    rows={2}
                    name="notes"
                    value={formData.notes}
                    onChange={handleInputChange}
                    placeholder="Enter notes (optional)"
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Check
                    type="checkbox"
                    name="isPrimary"
                    checked={formData.isPrimary}
                    onChange={handleInputChange}
                    label="Set as Primary"
                    className="mt-4"
                  />
                </Form.Group>
              </Col>
            </Row>

            <div className="d-flex gap-2">
              <Button 
                type="submit" 
                className="btn-bkash"
                disabled={loading}
              >
                {loading ? <Spinner animation="border" size="sm" /> : 'Update Account Number'}
              </Button>
              <Button 
                variant="outline-secondary"
                onClick={() => setShowEditModal(false)}
              >
                Cancel
              </Button>
            </div>
          </Form>
        </Modal.Body>
      </Modal>
    </div>
  );
};

// Account Numbers Table Component
const AccountNumbersTable = ({ 
  accounts, 
  loading, 
  onSetPrimary, 
  onToggleStatus, 
  onDelete,
  onEdit,
  getAccountTypeLabel,
  getAccountTypeBadge 
}) => {
  if (accounts.length === 0) {
    return (
      <Card>
        <Card.Body>
          <div className="empty-state text-center py-4">
            <i className="fas fa-credit-card fa-3x text-muted mb-3"></i>
            <h5>No Account Numbers</h5>
            <p className="text-muted">No account numbers found for this category.</p>
          </div>
        </Card.Body>
      </Card>
    );
  }

  return (
    <Card>
      <Card.Body>
        <div className="table-responsive">
          <Table striped hover>
            <thead>
              <tr>
                <th>Type</th>
                <th>Account Number</th>
                <th>Account Name</th>
                <th>Status</th>
                <th>Primary</th>
                <th>Created</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {accounts.map((account) => (
                <tr key={account.id}>
                  <td>
                    <Badge bg={getAccountTypeBadge(account.accountType)}>
                      {getAccountTypeLabel(account.accountType)}
                    </Badge>
                  </td>
                  <td>
                    <strong>{account.accountNumber}</strong>
                  </td>
                  <td>{account.accountName || '-'}</td>
                  <td>
                    <Badge bg={account.isActive ? 'success' : 'danger'}>
                      {account.isActive ? 'Active' : 'Inactive'}
                    </Badge>
                  </td>
                  <td>
                    {account.isPrimary ? (
                      <Badge bg="warning">
                        <i className="fas fa-star me-1"></i>
                        Primary
                      </Badge>
                    ) : (
                      <Button
                        variant="outline-warning"
                        size="sm"
                        onClick={() => onSetPrimary(account.id)}
                        disabled={loading}
                      >
                        Set Primary
                      </Button>
                    )}
                  </td>
                  <td>{new Date(account.createdAt).toLocaleDateString()}</td>
                  <td>
                    <div className="d-flex gap-1">
                      <Button
                        variant="outline-primary"
                        size="sm"
                        onClick={() => onEdit(account)}
                        disabled={loading}
                        title="Edit"
                      >
                        <i className="fas fa-edit"></i>
                      </Button>
                      <Button
                        variant={account.isActive ? 'outline-warning' : 'outline-success'}
                        size="sm"
                        onClick={() => onToggleStatus(account.id)}
                        disabled={loading}
                        title={account.isActive ? 'Deactivate' : 'Activate'}
                      >
                        <i className={`fas fa-${account.isActive ? 'pause' : 'play'}`}></i>
                      </Button>
                      <Button
                        variant="outline-danger"
                        size="sm"
                        onClick={() => onDelete(account.id)}
                        disabled={loading}
                        title="Delete"
                      >
                        <i className="fas fa-trash"></i>
                      </Button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </Table>
        </div>
      </Card.Body>
    </Card>
  );
};

export default BkashAccountNumbers;