import React, { useState, useRef } from 'react';
import { Container, Row, Col, Card, Button, Form, Alert, Table, Modal, Spinner } from 'react-bootstrap';
import { FaCamera, FaUpload, FaSave, FaEdit, FaTrash, FaEye } from 'react-icons/fa';
import { processOCRImage } from '../../services/ocrService';
import axios from 'axios';

const BkashOCRCapture = () => {
  const [selectedFile, setSelectedFile] = useState(null);
  const [previewUrl, setPreviewUrl] = useState(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [extractedData, setExtractedData] = useState(null);
  const [rawOCRText, setRawOCRText] = useState('');
  const [transactions, setTransactions] = useState([]);
  const [showModal, setShowModal] = useState(false);
  const [editingTransaction, setEditingTransaction] = useState(null);
  const [alert, setAlert] = useState({ show: false, message: '', variant: 'success' });
  const fileInputRef = useRef(null);

  // Form state for manual editing
  const [formData, setFormData] = useState({
    customer_number: '',
    transaction_id: '',
    amount: '',
    old_balance: '',
    new_balance: '',
    commission: '',
    transaction_type: 'Cash In',
    notes: ''
  });

  const showAlert = (message, variant = 'success') => {
    setAlert({ show: true, message, variant });
    setTimeout(() => setAlert({ show: false, message: '', variant: 'success' }), 5000);
  };

  const handleFileSelect = (event) => {
    const file = event.target.files[0];
    if (file) {
      setSelectedFile(file);
      const url = URL.createObjectURL(file);
      setPreviewUrl(url);
      setExtractedData(null);
      setRawOCRText('');
    }
  };

  const handleOCRProcess = async () => {
    if (!selectedFile) {
      showAlert('Please select an image first', 'warning');
      return;
    }

    setIsProcessing(true);
    try {
      const result = await processOCRImage(selectedFile);
      setRawOCRText(result.rawText);
      setExtractedData(result.extractedData);
      
      // Pre-fill form with extracted data
      setFormData({
        customer_number: result.extractedData.customerNumber || '',
        transaction_id: result.extractedData.transactionId || '',
        amount: result.extractedData.amount || '',
        old_balance: result.extractedData.oldBalance || '',
        new_balance: result.extractedData.newBalance || '',
        commission: result.extractedData.commission || '',
        transaction_type: result.extractedData.transactionType || 'Cash In',
        notes: ''
      });

      showAlert('OCR processing completed successfully!');
    } catch (error) {
      console.error('OCR processing error:', error);
      showAlert('Error processing image. Please try again.', 'danger');
    } finally {
      setIsProcessing(false);
    }
  };

  const handleFormChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSaveTransaction = async () => {
    try {
      const transactionData = {
        ...formData,
        raw_ocr_text: rawOCRText,
        ocr_confidence: extractedData?.confidence || 0,
        verification_status: 'pending'
      };

      const response = await axios.post('/api/recharge-transactions', transactionData);
      
      showAlert('Transaction saved successfully!');
      
      // Reset form and file selection
      setFormData({
        customer_number: '',
        transaction_id: '',
        amount: '',
        old_balance: '',
        new_balance: '',
        commission: '',
        transaction_type: 'Cash In',
        notes: ''
      });
      setSelectedFile(null);
      setPreviewUrl(null);
      setExtractedData(null);
      setRawOCRText('');
      
      // Refresh transactions list
      fetchTransactions();
      
    } catch (error) {
      console.error('Error saving transaction:', error);
      showAlert('Error saving transaction. Please try again.', 'danger');
    }
  };

  const fetchTransactions = async () => {
    try {
      const response = await axios.get('/api/recharge-transactions?limit=10');
      setTransactions(response.data.transactions || []);
    } catch (error) {
      console.error('Error fetching transactions:', error);
    }
  };

  const handleEditTransaction = (transaction) => {
    setEditingTransaction(transaction);
    setFormData({
      customer_number: transaction.customer_number || '',
      transaction_id: transaction.transaction_id || '',
      amount: transaction.amount || '',
      old_balance: transaction.old_balance || '',
      new_balance: transaction.new_balance || '',
      commission: transaction.commission || '',
      transaction_type: transaction.transaction_type || 'Cash In',
      notes: transaction.notes || ''
    });
    setShowModal(true);
  };

  const handleUpdateTransaction = async () => {
    try {
      await axios.put(`/api/recharge-transactions/${editingTransaction.id}`, formData);
      showAlert('Transaction updated successfully!');
      setShowModal(false);
      setEditingTransaction(null);
      fetchTransactions();
    } catch (error) {
      console.error('Error updating transaction:', error);
      showAlert('Error updating transaction. Please try again.', 'danger');
    }
  };

  const handleDeleteTransaction = async (id) => {
    if (window.confirm('Are you sure you want to delete this transaction?')) {
      try {
        await axios.delete(`/api/recharge-transactions/${id}`);
        showAlert('Transaction deleted successfully!');
        fetchTransactions();
      } catch (error) {
        console.error('Error deleting transaction:', error);
        showAlert('Error deleting transaction. Please try again.', 'danger');
      }
    }
  };

  const handleVerifyTransaction = async (id) => {
    try {
      await axios.put(`/api/recharge-transactions/${id}/verify`);
      showAlert('Transaction verified successfully!');
      fetchTransactions();
    } catch (error) {
      console.error('Error verifying transaction:', error);
      showAlert('Error verifying transaction. Please try again.', 'danger');
    }
  };

  React.useEffect(() => {
    fetchTransactions();
  }, []);

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h2>bKash Agent Auto Transaction Entry</h2>
          <p className="text-muted">Capture bKash confirmation screens and automatically extract transaction data</p>
        </Col>
      </Row>

      {alert.show && (
        <Alert variant={alert.variant} dismissible onClose={() => setAlert({ show: false, message: '', variant: 'success' })}>
          {alert.message}
        </Alert>
      )}

      <Row>
        <Col lg={6}>
          <Card className="mb-4">
            <Card.Header>
              <h5><FaCamera className="me-2" />Image Capture & OCR</h5>
            </Card.Header>
            <Card.Body>
              <div className="mb-3">
                <Form.Group>
                  <Form.Label>Select bKash Confirmation Screen</Form.Label>
                  <Form.Control
                    type="file"
                    accept="image/*"
                    onChange={handleFileSelect}
                    ref={fileInputRef}
                  />
                </Form.Group>
              </div>

              {previewUrl && (
                <div className="mb-3">
                  <img 
                    src={previewUrl} 
                    alt="Preview" 
                    style={{ maxWidth: '100%', maxHeight: '300px', objectFit: 'contain' }}
                    className="border rounded"
                  />
                </div>
              )}

              <div className="d-grid gap-2">
                <Button 
                  variant="primary" 
                  onClick={handleOCRProcess}
                  disabled={!selectedFile || isProcessing}
                >
                  {isProcessing ? (
                    <>
                      <Spinner animation="border" size="sm" className="me-2" />
                      Processing OCR...
                    </>
                  ) : (
                    <>
                      <FaUpload className="me-2" />
                      Process with OCR
                    </>
                  )}
                </Button>
              </div>

              {rawOCRText && (
                <div className="mt-3">
                  <Form.Group>
                    <Form.Label>Raw OCR Text</Form.Label>
                    <Form.Control
                      as="textarea"
                      rows={4}
                      value={rawOCRText}
                      readOnly
                      className="small"
                    />
                  </Form.Group>
                </div>
              )}
            </Card.Body>
          </Card>
        </Col>

        <Col lg={6}>
          <Card className="mb-4">
            <Card.Header>
              <h5><FaEdit className="me-2" />Transaction Data</h5>
            </Card.Header>
            <Card.Body>
              <Form>
                <Row>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Customer Number</Form.Label>
                      <Form.Control
                        type="text"
                        name="customer_number"
                        value={formData.customer_number}
                        onChange={handleFormChange}
                        placeholder="01XXXXXXXXX"
                      />
                    </Form.Group>
                  </Col>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Transaction ID</Form.Label>
                      <Form.Control
                        type="text"
                        name="transaction_id"
                        value={formData.transaction_id}
                        onChange={handleFormChange}
                        placeholder="Transaction ID"
                      />
                    </Form.Group>
                  </Col>
                </Row>

                <Row>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Amount</Form.Label>
                      <Form.Control
                        type="number"
                        step="0.01"
                        name="amount"
                        value={formData.amount}
                        onChange={handleFormChange}
                        placeholder="0.00"
                      />
                    </Form.Group>
                  </Col>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Commission</Form.Label>
                      <Form.Control
                        type="number"
                        step="0.01"
                        name="commission"
                        value={formData.commission}
                        onChange={handleFormChange}
                        placeholder="0.00"
                      />
                    </Form.Group>
                  </Col>
                </Row>

                <Row>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Old Balance</Form.Label>
                      <Form.Control
                        type="number"
                        step="0.01"
                        name="old_balance"
                        value={formData.old_balance}
                        onChange={handleFormChange}
                        placeholder="0.00"
                      />
                    </Form.Group>
                  </Col>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>New Balance</Form.Label>
                      <Form.Control
                        type="number"
                        step="0.01"
                        name="new_balance"
                        value={formData.new_balance}
                        onChange={handleFormChange}
                        placeholder="0.00"
                      />
                    </Form.Group>
                  </Col>
                </Row>

                <Row>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Transaction Type</Form.Label>
                      <Form.Select
                        name="transaction_type"
                        value={formData.transaction_type}
                        onChange={handleFormChange}
                      >
                        <option value="Cash In">Cash In</option>
                        <option value="Cash Out">Cash Out</option>
                        <option value="Mobile Recharge">Mobile Recharge</option>
                        <option value="Payment">Payment</option>
                        <option value="Other">Other</option>
                      </Form.Select>
                    </Form.Group>
                  </Col>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label>Notes</Form.Label>
                      <Form.Control
                        type="text"
                        name="notes"
                        value={formData.notes}
                        onChange={handleFormChange}
                        placeholder="Optional notes"
                      />
                    </Form.Group>
                  </Col>
                </Row>

                <div className="d-grid">
                  <Button 
                    variant="success" 
                    onClick={handleSaveTransaction}
                    disabled={!formData.customer_number || !formData.transaction_id || !formData.amount}
                  >
                    <FaSave className="me-2" />
                    Save Transaction
                  </Button>
                </div>
              </Form>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      <Row>
        <Col>
          <Card>
            <Card.Header>
              <h5>Recent Transactions</h5>
            </Card.Header>
            <Card.Body>
              <Table responsive striped hover>
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>Customer</th>
                    <th>Transaction ID</th>
                    <th>Type</th>
                    <th>Amount</th>
                    <th>Commission</th>
                    <th>Status</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {transactions.map((transaction) => (
                    <tr key={transaction.id}>
                      <td>{new Date(transaction.created_at).toLocaleDateString()}</td>
                      <td>{transaction.customer_number}</td>
                      <td>{transaction.transaction_id}</td>
                      <td>{transaction.transaction_type}</td>
                      <td>৳{parseFloat(transaction.amount || 0).toFixed(2)}</td>
                      <td>৳{parseFloat(transaction.commission || 0).toFixed(2)}</td>
                      <td>
                        <span className={`badge ${transaction.verification_status === 'verified' ? 'bg-success' : 'bg-warning'}`}>
                          {transaction.verification_status}
                        </span>
                      </td>
                      <td>
                        <Button
                          variant="outline-primary"
                          size="sm"
                          className="me-1"
                          onClick={() => handleEditTransaction(transaction)}
                        >
                          <FaEdit />
                        </Button>
                        {transaction.verification_status !== 'verified' && (
                          <Button
                            variant="outline-success"
                            size="sm"
                            className="me-1"
                            onClick={() => handleVerifyTransaction(transaction.id)}
                          >
                            <FaEye />
                          </Button>
                        )}
                        <Button
                          variant="outline-danger"
                          size="sm"
                          onClick={() => handleDeleteTransaction(transaction.id)}
                        >
                          <FaTrash />
                        </Button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Edit Modal */}
      <Modal show={showModal} onHide={() => setShowModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>Edit Transaction</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Customer Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="customer_number"
                    value={formData.customer_number}
                    onChange={handleFormChange}
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Transaction ID</Form.Label>
                  <Form.Control
                    type="text"
                    name="transaction_id"
                    value={formData.transaction_id}
                    onChange={handleFormChange}
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Amount</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="amount"
                    value={formData.amount}
                    onChange={handleFormChange}
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Commission</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="commission"
                    value={formData.commission}
                    onChange={handleFormChange}
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Transaction Type</Form.Label>
                  <Form.Select
                    name="transaction_type"
                    value={formData.transaction_type}
                    onChange={handleFormChange}
                  >
                    <option value="Cash In">Cash In</option>
                    <option value="Cash Out">Cash Out</option>
                    <option value="Mobile Recharge">Mobile Recharge</option>
                    <option value="Payment">Payment</option>
                    <option value="Other">Other</option>
                  </Form.Select>
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Notes</Form.Label>
                  <Form.Control
                    type="text"
                    name="notes"
                    value={formData.notes}
                    onChange={handleFormChange}
                  />
                </Form.Group>
              </Col>
            </Row>
          </Form>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowModal(false)}>
            Cancel
          </Button>
          <Button variant="primary" onClick={handleUpdateTransaction}>
            Update Transaction
          </Button>
        </Modal.Footer>
      </Modal>
    </Container>
  );
};

export default BkashOCRCapture;