import React, { useState, useEffect } from 'react';
import { Row, Col, Card, Form, Button, Table, Badge, Spinner, Modal, Tab, Tabs } from 'react-bootstrap';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';

const BkashPersonalAccount = ({ showAlert }) => {
  const { currentUser } = useAuth();
  const [accountData, setAccountData] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('account');
  const [accountNumbers, setAccountNumbers] = useState([]);
  const [selectedAccount, setSelectedAccount] = useState(null);
  const [formData, setFormData] = useState({
    openingBalance: '',
    closingBalance: '',
    date: new Date().toISOString().split('T')[0]
  });
  const [transactionForm, setTransactionForm] = useState({
    type: 'bank_transaction',
    amount: '',
    description: '',
    date: new Date().toISOString().split('T')[0]
  });
  const [showCloseModal, setShowCloseModal] = useState(false);
  const [summary, setSummary] = useState(null);

  useEffect(() => {
    // Only fetch data if user is authenticated
    if (currentUser) {
      fetchAccountNumbers();
    }
  }, [currentUser]);

  useEffect(() => {
    // Fetch account data when a personal account is selected
    if (selectedAccount) {
      fetchTodayAccount();
      fetchTodayTransactions();
      fetchSummary();
    }
  }, [selectedAccount]);

  const fetchAccountNumbers = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-account-numbers?accountType=personal', {
        headers: { Authorization: `Bearer ${token}` }
      });
      const personalAccounts = response.data.filter(account => account.isActive);
      setAccountNumbers(personalAccounts);
      
      // Auto-select the first primary personal account or first personal account
      const primaryAccount = personalAccounts.find(account => account.isPrimary);
      if (primaryAccount) {
        setSelectedAccount(primaryAccount);
      } else if (personalAccounts.length > 0) {
        setSelectedAccount(personalAccounts[0]);
      }
    } catch (error) {
      console.error('Error fetching account numbers:', error);
      showAlert('Error fetching personal account numbers', 'danger');
    }
  };

  const fetchTodayAccount = async () => {
    if (!selectedAccount) return;
    
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-personal/today', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAccountData(response.data);
      if (response.data) {
        setFormData({
          openingBalance: response.data.openingBalance || '',
          closingBalance: response.data.closingBalance || '',
          date: response.data.date || new Date().toISOString().split('T')[0]
        });
      }
    } catch (error) {
      console.error('Error fetching today account:', error);
      showAlert('Error fetching account data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchTodayTransactions = async () => {
    if (!selectedAccount) return;
    
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const today = new Date().toISOString().split('T')[0];
      const response = await axios.get(`http://localhost:5000/api/bkash-personal/${today}/transactions`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTransactions(response.data);
    } catch (error) {
      console.error('Error fetching transactions:', error);
      showAlert('Error fetching transactions', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchSummary = async () => {
    if (!selectedAccount) return;
    
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-personal/summary', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setSummary(response.data);
    } catch (error) {
      console.error('Error fetching summary:', error);
    }
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleTransactionChange = (e) => {
    const { name, value } = e.target;
    setTransactionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleAccountSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...formData,
        openingBalance: parseFloat(formData.openingBalance) || 0,
        closingBalance: parseFloat(formData.closingBalance) || 0
      };

      await axios.post('http://localhost:5000/api/bkash-personal', submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Personal account updated successfully!', 'success');
      fetchTodayAccount();
      fetchSummary();
    } catch (error) {
      console.error('Error updating account:', error);
      showAlert('Error updating account', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleTransactionSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...transactionForm,
        amount: parseFloat(transactionForm.amount) || 0
      };

      await axios.post('http://localhost:5000/api/bkash-personal/transaction', submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Transaction added successfully!', 'success');
      setTransactionForm({
        type: 'bank_transaction',
        amount: '',
        description: '',
        date: new Date().toISOString().split('T')[0]
      });
      fetchTodayTransactions();
      fetchTodayAccount();
    } catch (error) {
      console.error('Error adding transaction:', error);
      showAlert('Error adding transaction', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleCloseDay = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.put('http://localhost:5000/api/bkash-personal/close', {
        date: formData.date,
        closingBalance: parseFloat(formData.closingBalance) || 0
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Day closed successfully!', 'success');
      setShowCloseModal(false);
      fetchTodayAccount();
      fetchSummary();
    } catch (error) {
      console.error('Error closing day:', error);
      showAlert('Error closing day', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const calculateBalance = () => {
    if (!accountData) return 0;
    const opening = parseFloat(accountData.openingBalance) || 0;
    const closing = parseFloat(formData.closingBalance) || parseFloat(accountData.closingBalance) || 0;
    return closing - opening;
  };

  const getTransactionTypeLabel = (type) => {
    switch (type) {
      case 'sr_transaction': return 'SR Transaction';
      case 'bank_transaction': return 'Bank Transaction';
      case 'other': return 'Other';
      default: return type;
    }
  };

  const getTransactionTypeBadge = (type) => {
    switch (type) {
      case 'sr_transaction': return 'primary';
      case 'bank_transaction': return 'success';
      case 'other': return 'secondary';
      default: return 'secondary';
    }
  };

  if (loading && !accountData) {
    return (
      <div className="loading-spinner">
        <Spinner animation="border" variant="primary" />
      </div>
    );
  }

  return (
    <div>
      {/* Summary Cards */}
      {summary && (
        <Row className="mb-4">
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.totalBalance?.toLocaleString() || '0'}</div>
                <div className="summary-label">Total Balance</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">{summary.totalTransactions || '0'}</div>
                <div className="summary-label">Total Transactions</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.avgBalance?.toLocaleString() || '0'}</div>
                <div className="summary-label">Avg Daily Balance</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.lastClosing?.toLocaleString() || '0'}</div>
                <div className="summary-label">Last Closing</div>
              </Card.Body>
            </Card>
          </Col>
        </Row>
      )}

      {/* Account Selection */}
      <Row className="mb-4">
        <Col md={6}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">
                <i className="fas fa-user me-2"></i>
                Select Personal Account
              </h5>
            </Card.Header>
            <Card.Body>
              <Form.Group>
                <Form.Label>Personal Account Number</Form.Label>
                <Form.Select
                  value={selectedAccount?.id || ''}
                  onChange={(e) => {
                    const accountId = parseInt(e.target.value);
                    const account = accountNumbers.find(acc => acc.id === accountId);
                    setSelectedAccount(account);
                  }}
                  disabled={loading}
                >
                  <option value="">Select a personal account...</option>
                  {accountNumbers.map(account => (
                    <option key={account.id} value={account.id}>
                      {account.accountNumber} - {account.accountName || 'Personal Account'}
                      {account.isPrimary ? ' (Primary)' : ''}
                    </option>
                  ))}
                </Form.Select>
              </Form.Group>
              
              {selectedAccount && (
                <div className="mt-3 p-3 bg-light rounded">
                  <div className="d-flex justify-content-between align-items-center">
                    <div>
                      <h6 className="mb-1">
                        <Badge bg="info" className="me-2">Personal</Badge>
                        {selectedAccount.accountNumber}
                      </h6>
                      <small className="text-muted">{selectedAccount.accountName}</small>
                    </div>
                    <div className="d-flex gap-2">
                      <Button
                        variant="outline-primary"
                        size="sm"
                        onClick={() => window.open('/bkash/account-numbers', '_blank')}
                      >
                        <i className="fas fa-edit me-1"></i>
                        Edit Account
                      </Button>
                    </div>
                  </div>
                </div>
              )}
            </Card.Body>
          </Card>
        </Col>
        
        {accountNumbers.length === 0 && !loading && (
          <Col md={6}>
            <Card className="text-center">
              <Card.Body className="py-5">
                <i className="fas fa-mobile-alt fa-3x text-muted mb-3"></i>
                <h5>No Personal Accounts Found</h5>
                <p className="text-muted mb-4">You need to add a personal bKash account number first.</p>
                <Button
                  variant="primary"
                  onClick={() => window.open('/bkash/account-numbers', '_blank')}
                >
                  <i className="fas fa-plus me-2"></i>
                  Add Personal Account
                </Button>
              </Card.Body>
            </Card>
          </Col>
        )}
      </Row>

      {/* Main Content Tabs */}
      {selectedAccount ? (
        <Tabs activeKey={activeTab} onSelect={(k) => setActiveTab(k)} className="mb-3">
          <Tab eventKey="account" title="Account Management">
          <Row>
            <Col lg={6}>
              <Card className="account-card">
                <div className="account-header">
                  <h5 className="mb-0">
                    <i className="fas fa-user me-2"></i>
                    Personal Account - {formData.date}
                  </h5>
                </div>
                <Card.Body>
                  <Form onSubmit={handleAccountSubmit}>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Date</Form.Label>
                          <Form.Control
                            type="date"
                            name="date"
                            value={formData.date}
                            onChange={handleInputChange}
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Opening Balance (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="openingBalance"
                            value={formData.openingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter opening balance"
                            required
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Closing Balance (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="closingBalance"
                            value={formData.closingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter closing balance"
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Net Change</Form.Label>
                          <div className="profit-display">
                            ৳{calculateBalance().toLocaleString()}
                          </div>
                        </Form.Group>
                      </Col>
                    </Row>

                    <div className="d-flex gap-2">
                      <Button 
                        type="submit" 
                        className="btn-bkash"
                        disabled={loading}
                      >
                        {loading ? <Spinner animation="border" size="sm" /> : 'Save Account'}
                      </Button>
                      
                      {accountData && formData.closingBalance && (
                        <Button 
                          variant="outline-success"
                          onClick={() => setShowCloseModal(true)}
                          disabled={loading}
                        >
                          Close Day
                        </Button>
                      )}
                    </div>
                  </Form>
                </Card.Body>
              </Card>
            </Col>

            <Col lg={6}>
              <Card className="account-card">
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-chart-line me-2"></i>
                    Account Status
                  </h6>
                </Card.Header>
                <Card.Body>
                  {accountData ? (
                    <div>
                      <div className="mb-3">
                        <strong>Opening Balance:</strong> ৳{accountData.openingBalance?.toLocaleString() || '0'}
                      </div>
                      <div className="mb-3">
                        <strong>Current Closing:</strong> ৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}
                      </div>
                      <div className="mb-3">
                        <strong>Net Change:</strong> 
                        <span className={calculateBalance() >= 0 ? 'text-success' : 'text-danger'}>
                          ৳{calculateBalance().toLocaleString()}
                        </span>
                      </div>
                      <div className="mb-3">
                        <strong>Status:</strong> 
                        <Badge bg={accountData.isClosed ? 'success' : 'warning'} className="ms-2">
                          {accountData.isClosed ? 'Closed' : 'Open'}
                        </Badge>
                      </div>
                    </div>
                  ) : (
                    <div className="empty-state">
                      <i className="fas fa-plus-circle"></i>
                      <p>No account data for today. Create a new entry above.</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </Tab>

        <Tab eventKey="transactions" title="Transactions">
          <Row>
            <Col lg={6}>
              <Card className="transaction-form">
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-plus me-2"></i>
                    Add Transaction
                  </h6>
                </Card.Header>
                <Card.Body>
                  <Form onSubmit={handleTransactionSubmit}>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Transaction Type</Form.Label>
                          <Form.Select
                            name="type"
                            value={transactionForm.type}
                            onChange={handleTransactionChange}
                            required
                          >
                            <option value="bank_transaction">Bank Transaction</option>
                            <option value="other">Other</option>
                          </Form.Select>
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Amount (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="amount"
                            value={transactionForm.amount}
                            onChange={handleTransactionChange}
                            placeholder="Enter amount"
                            required
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Date</Form.Label>
                          <Form.Control
                            type="date"
                            name="date"
                            value={transactionForm.date}
                            onChange={handleTransactionChange}
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Description</Form.Label>
                          <Form.Control
                            type="text"
                            name="description"
                            value={transactionForm.description}
                            onChange={handleTransactionChange}
                            placeholder="Enter description"
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Button 
                      type="submit" 
                      className="btn-bkash"
                      disabled={loading}
                    >
                      {loading ? <Spinner animation="border" size="sm" /> : 'Add Transaction'}
                    </Button>
                  </Form>
                </Card.Body>
              </Card>
            </Col>

            <Col lg={6}>
              <Card>
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-list me-2"></i>
                    Today's Transactions
                  </h6>
                </Card.Header>
                <Card.Body>
                  {transactions.length > 0 ? (
                    <Table responsive size="sm" className="transaction-table">
                      <thead>
                        <tr>
                          <th>Type</th>
                          <th>Amount</th>
                          <th>Description</th>
                        </tr>
                      </thead>
                      <tbody>
                        {transactions.map((transaction, index) => (
                          <tr key={index} className="transaction-row">
                            <td>
                              <Badge bg={getTransactionTypeBadge(transaction.type)}>
                                {getTransactionTypeLabel(transaction.type)}
                              </Badge>
                            </td>
                            <td>৳{transaction.amount?.toLocaleString()}</td>
                            <td>{transaction.description || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </Table>
                  ) : (
                    <div className="empty-state">
                      <i className="fas fa-receipt"></i>
                      <p>No transactions for today</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </Tab>


      </Tabs>
      ) : (
        <Row className="justify-content-center">
          <Col md={6}>
            <Card className="text-center">
              <Card.Body>
                <i className="fas fa-mobile-alt fa-3x text-muted mb-3"></i>
                <h5>No Personal Account Selected</h5>
                <p className="text-muted">
                  Please select a personal account from the dropdown above or add a new one.
                </p>
                <Button 
                  variant="primary" 
                  onClick={() => window.location.href = '/bkash/account-numbers'}
                >
                  <i className="fas fa-plus me-2"></i>
                  Add Personal Account
                </Button>
              </Card.Body>
            </Card>
          </Col>
        </Row>
      )}

      {/* Close Day Modal */}
      <Modal show={showCloseModal} onHide={() => setShowCloseModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Close Day</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <p>Are you sure you want to close the day with the following details?</p>
          <ul>
            <li><strong>Date:</strong> {formData.date}</li>
            <li><strong>Opening Balance:</strong> ৳{(parseFloat(formData.openingBalance) || 0).toLocaleString()}</li>
            <li><strong>Closing Balance:</strong> ৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}</li>
            <li><strong>Net Change:</strong> ৳{calculateBalance().toLocaleString()}</li>
            <li><strong>Transactions:</strong> {transactions.length}</li>
          </ul>
          <p className="text-warning">
            <i className="fas fa-exclamation-triangle me-2"></i>
            This action cannot be undone.
          </p>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowCloseModal(false)}>
            Cancel
          </Button>
          <Button variant="success" onClick={handleCloseDay} disabled={loading}>
            {loading ? <Spinner animation="border" size="sm" /> : 'Close Day'}
          </Button>
        </Modal.Footer>
      </Modal>
    </div>
  );
};

export default BkashPersonalAccount;