import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Table, Badge, Button, Alert, Spinner, Form, InputGroup } from 'react-bootstrap';
import { FaSync, FaSearch, FaFilter, FaMobileAlt, FaMoneyBillWave } from 'react-icons/fa';
import bkashSmsService from '../../services/bkashSmsService';
import './BkashSmsTransactions.css';

const BkashSmsTransactions = () => {
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, message: '', variant: 'success' });
  const [searchTerm, setSearchTerm] = useState('');
  const [filterType, setFilterType] = useState('all');
  const [serverStatus, setServerStatus] = useState('checking');

  const showAlert = (message, type = 'success') => {
    setAlert({ show: true, message, variant: type });
    setTimeout(() => setAlert({ show: false, message: '', variant: 'success' }), 5000);
  };

  const checkServerStatus = async () => {
    const status = await bkashSmsService.getConnectionStatus();
    setServerStatus(status.connected ? 'connected' : 'disconnected');
  };

  const fetchTransactions = async () => {
    setLoading(true);
    try {
      const result = await bkashSmsService.getTransactions();
      if (result.success) {
        setTransactions(result.data.transactions || []);
        showAlert(`Loaded ${result.data.transactions?.length || 0} SMS transactions`, 'success');
      } else {
        throw result.error;
      }
    } catch (error) {
      console.error('Error fetching transactions:', error);
      showAlert(error.message || 'Failed to fetch SMS transactions. Make sure the SMS server is running.', 'danger');
      setTransactions([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    checkServerStatus();
    fetchTransactions();
    
    // Auto-refresh every 30 seconds
    const interval = setInterval(() => {
      fetchTransactions();
    }, 30000);

    return () => clearInterval(interval);
  }, []);

  const getStatusBadge = (status) => {
    const statusMap = {
      'pending': 'warning',
      'processed': 'success',
      'failed': 'danger',
      'duplicate': 'secondary'
    };
    return <Badge bg={statusMap[status] || 'secondary'}>{status}</Badge>;
  };

  const getTransactionTypeBadge = (type) => {
    const typeMap = {
      'cash_in': 'success',
      'cash_out': 'danger',
      'send_money': 'primary',
      'payment': 'info',
      'mobile_recharge': 'warning'
    };
    return <Badge bg={typeMap[type] || 'secondary'}>{type?.replace('_', ' ')}</Badge>;
  };

  const filteredTransactions = transactions.filter(transaction => {
    const matchesSearch = transaction.sender_number?.includes(searchTerm) ||
                         transaction.transaction_id?.includes(searchTerm) ||
                         transaction.amount?.toString().includes(searchTerm);
    
    const matchesFilter = filterType === 'all' || transaction.transaction_type === filterType;
    
    return matchesSearch && matchesFilter;
  });

  const formatCurrency = (amount) => {
    return bkashSmsService.formatCurrency(amount);
  };

  const formatDate = (dateString) => {
    return bkashSmsService.formatDate(dateString);
  };

  const getServerStatusBadge = () => {
    const statusMap = {
      'checking': { variant: 'secondary', text: 'Checking...', icon: 'fa-spinner fa-spin' },
      'connected': { variant: 'success', text: 'Connected', icon: 'fa-check-circle' },
      'disconnected': { variant: 'danger', text: 'Disconnected', icon: 'fa-times-circle' },
      'error': { variant: 'warning', text: 'Error', icon: 'fa-exclamation-triangle' }
    };
    
    const status = statusMap[serverStatus];
    return (
      <Badge bg={status.variant} className="me-2">
        <i className={`fas ${status.icon} me-1`}></i>
        SMS Server: {status.text}
      </Badge>
    );
  };

  return (
    <Container fluid className="bkash-sms-transactions">
      <Row className="mb-4">
        <Col>
          <div className="d-flex justify-content-between align-items-center">
            <div>
              <h2 className="page-title">
                <FaMobileAlt className="me-2" />
                bKash SMS Transactions
              </h2>
              <p className="text-muted">Real-time SMS transaction monitoring from Android device</p>
            </div>
            <div>
              {getServerStatusBadge()}
              <Button 
                variant="primary" 
                onClick={fetchTransactions}
                disabled={loading}
                className="ms-2"
              >
                {loading ? <Spinner size="sm" className="me-1" /> : <FaSync className="me-1" />}
                Refresh
              </Button>
            </div>
          </div>
        </Col>
      </Row>

      {alert.show && (
        <Row className="mb-3">
          <Col>
            <Alert variant={alert.variant} dismissible onClose={() => setAlert({ show: false, message: '', variant: 'success' })}>
              {alert.message}
            </Alert>
          </Col>
        </Row>
      )}

      <Row className="mb-3">
        <Col md={6}>
          <InputGroup>
            <InputGroup.Text><FaSearch /></InputGroup.Text>
            <Form.Control
              type="text"
              placeholder="Search by phone number, transaction ID, or amount..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
          </InputGroup>
        </Col>
        <Col md={3}>
          <InputGroup>
            <InputGroup.Text><FaFilter /></InputGroup.Text>
            <Form.Select
              value={filterType}
              onChange={(e) => setFilterType(e.target.value)}
            >
              <option value="all">All Types</option>
              <option value="cash_in">Cash In</option>
              <option value="cash_out">Cash Out</option>
              <option value="send_money">Send Money</option>
              <option value="payment">Payment</option>
              <option value="mobile_recharge">Mobile Recharge</option>
            </Form.Select>
          </InputGroup>
        </Col>
        <Col md={3}>
          <div className="text-end">
            <strong>Total: {filteredTransactions.length} transactions</strong>
          </div>
        </Col>
      </Row>

      <Row>
        <Col>
          <Card className="shadow-sm">
            <Card.Header className="bg-primary text-white">
              <h5 className="mb-0">
                <FaMoneyBillWave className="me-2" />
                SMS Transaction History
              </h5>
            </Card.Header>
            <Card.Body className="p-0">
              {loading ? (
                <div className="text-center p-4">
                  <Spinner animation="border" role="status">
                    <span className="visually-hidden">Loading...</span>
                  </Spinner>
                  <p className="mt-2">Loading transactions...</p>
                </div>
              ) : filteredTransactions.length === 0 ? (
                <div className="text-center p-4">
                  <FaMobileAlt size={48} className="text-muted mb-3" />
                  <h5>No SMS transactions found</h5>
                  <p className="text-muted">
                    {transactions.length === 0 
                      ? "No SMS transactions have been received yet. Make sure your Android app is configured and sending SMS data."
                      : "No transactions match your current search criteria."
                    }
                  </p>
                </div>
              ) : (
                <Table responsive striped hover className="mb-0">
                  <thead className="table-dark">
                    <tr>
                      <th>Date & Time</th>
                      <th>Transaction ID</th>
                      <th>Type</th>
                      <th>Sender</th>
                      <th>Amount</th>
                      <th>Balance</th>
                      <th>Status</th>
                      <th>Raw SMS</th>
                    </tr>
                  </thead>
                  <tbody>
                    {filteredTransactions.map((transaction, index) => (
                      <tr key={transaction.id || index}>
                        <td>{formatDate(transaction.created_at)}</td>
                        <td>
                          <code className="text-primary">{transaction.transaction_id}</code>
                        </td>
                        <td>{getTransactionTypeBadge(transaction.transaction_type)}</td>
                        <td>{transaction.sender_number}</td>
                        <td className="fw-bold text-success">
                          {formatCurrency(transaction.amount)}
                        </td>
                        <td>{transaction.balance ? formatCurrency(transaction.balance) : 'N/A'}</td>
                        <td>{getStatusBadge(transaction.status)}</td>
                        <td>
                          <small className="text-muted" title={transaction.raw_sms}>
                            {transaction.raw_sms?.substring(0, 50)}...
                          </small>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </Table>
              )}
            </Card.Body>
          </Card>
        </Col>
      </Row>

      <Row className="mt-3">
        <Col>
          <Card className="bg-light">
            <Card.Body>
              <h6>📱 SMS Integration Status</h6>
              <p className="mb-1">
                <strong>SMS Server:</strong> {bkashSmsService.baseURL} {getServerStatusBadge()}
              </p>
              <p className="mb-1">
                <strong>Auto-refresh:</strong> Every 30 seconds
              </p>
              <p className="mb-0">
                <strong>Last updated:</strong> {new Date().toLocaleTimeString()}
              </p>
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default BkashSmsTransactions;