import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Table, Form, Button, Alert, Spinner, Badge } from 'react-bootstrap';
import axios from 'axios';
import './CashBox.css';

const CashBox = () => {
  const [reconciliation, setReconciliation] = useState(null);
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [alert, setAlert] = useState({ show: false, message: '', type: '' });

  const noteValues = [5, 10, 20, 50, 100, 200, 500, 1000];

  useEffect(() => {
    fetchTodayReconciliation();
  }, []);

  const fetchTodayReconciliation = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/cash-reconciliation/today', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setReconciliation(response.data);
    } catch (error) {
      console.error('Error fetching reconciliation:', error);
      showAlert('Error fetching cash reconciliation data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const showAlert = (message, type) => {
    setAlert({ show: true, message, type });
    setTimeout(() => setAlert({ show: false, message: '', type: '' }), 5000);
  };

  const handleNoteChange = (noteValue, type, value) => {
    const fieldName = `${type}_${noteValue}_notes`;
    setReconciliation(prev => ({
      ...prev,
      [fieldName]: parseInt(value) || 0
    }));
  };

  const calculateTotal = (type) => {
    if (!reconciliation) return 0;
    return noteValues.reduce((total, noteValue) => {
      const count = reconciliation[`${type}_${noteValue}_notes`] || 0;
      return total + (count * noteValue);
    }, 0);
  };

  const calculateVariance = () => {
    if (!reconciliation) return 0;
    const openingTotal = calculateTotal('opening');
    const closingTotal = calculateTotal('closing');
    const totalIncome = parseFloat(reconciliation.total_income || 0);
    const totalExpense = parseFloat(reconciliation.total_expense || 0);
    
    const expectedClosing = openingTotal + totalIncome - totalExpense;
    return closingTotal - expectedClosing;
  };

  const saveReconciliation = async () => {
    try {
      setSaving(true);
      const token = localStorage.getItem('token');
      const today = new Date().toISOString().split('T')[0];
      
      const updateData = {
        ...reconciliation,
        total_income: parseFloat(reconciliation.total_income || 0),
        total_expense: parseFloat(reconciliation.total_expense || 0)
      };

      await axios.put(`http://localhost:5000/api/cash-reconciliation/${today}`, updateData, {
        headers: { Authorization: `Bearer ${token}` }
      });

      showAlert('Cash reconciliation saved successfully!', 'success');
      fetchTodayReconciliation(); // Refresh data
    } catch (error) {
      console.error('Error saving reconciliation:', error);
      showAlert('Error saving cash reconciliation', 'danger');
    } finally {
      setSaving(false);
    }
  };

  const getVarianceColor = () => {
    const variance = calculateVariance();
    if (variance === 0) return 'success';
    if (variance < 0) return 'danger';
    return 'warning';
  };

  if (loading) {
    return (
      <Container className="mt-4 text-center">
        <Spinner animation="border" role="status">
          <span className="visually-hidden">Loading...</span>
        </Spinner>
      </Container>
    );
  }

  return (
    <Container fluid className="cash-box-container">
      <Row className="mb-4">
        <Col>
          <h2 className="page-title">
            <i className="fas fa-cash-register me-2"></i>
            Cash Box - Daily Cash Reconciliation
          </h2>
          <p className="text-muted">Track opening and closing cash counts for daily reconciliation</p>
        </Col>
      </Row>

      {alert.show && (
        <Alert variant={alert.type} dismissible onClose={() => setAlert({ show: false, message: '', type: '' })}>
          {alert.message}
        </Alert>
      )}

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card opening-card">
            <Card.Body className="text-center">
              <h5 className="card-title">Opening Cash</h5>
              <h3 className="card-amount">৳{calculateTotal('opening').toLocaleString()}</h3>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card closing-card">
            <Card.Body className="text-center">
              <h5 className="card-title">Closing Cash</h5>
              <h3 className="card-amount">৳{calculateTotal('closing').toLocaleString()}</h3>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card variance-card">
            <Card.Body className="text-center">
              <h5 className="card-title">Daily Variance</h5>
              <h3 className={`card-amount text-${getVarianceColor()}`}>
                ৳{calculateVariance().toLocaleString()}
              </h3>
              <Badge bg={getVarianceColor()}>
                {calculateVariance() === 0 ? 'Balanced' : calculateVariance() < 0 ? 'Short' : 'Over'}
              </Badge>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card income-expense-card">
            <Card.Body>
              <Row>
                <Col>
                  <Form.Group className="mb-2">
                    <Form.Label className="small">Total Income</Form.Label>
                    <Form.Control
                      type="number"
                      step="0.01"
                      value={reconciliation?.total_income || ''}
                      onChange={(e) => setReconciliation(prev => ({
                        ...prev,
                        total_income: e.target.value
                      }))}
                      placeholder="0.00"
                    />
                  </Form.Group>
                  <Form.Group>
                    <Form.Label className="small">Total Expense</Form.Label>
                    <Form.Control
                      type="number"
                      step="0.01"
                      value={reconciliation?.total_expense || ''}
                      onChange={(e) => setReconciliation(prev => ({
                        ...prev,
                        total_expense: e.target.value
                      }))}
                      placeholder="0.00"
                    />
                  </Form.Group>
                </Col>
              </Row>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Note Tracking Table */}
      <Row>
        <Col>
          <Card className="notes-table-card">
            <Card.Header>
              <h5 className="mb-0">
                <i className="fas fa-table me-2"></i>
                Note Tracking Table
              </h5>
              <small className="text-muted">
                <i className="fas fa-arrows-alt-h me-1"></i>
                Scroll horizontally to see all denominations (5tk - 1000tk)
              </small>
            </Card.Header>
            <Card.Body>
                
                <div className="table-responsive">
                <Table className="notes-table" striped bordered hover>
                  <thead>
                    <tr>
                      <th>Note Value</th>
                      <th>Opening Count</th>
                      <th>Opening Total</th>
                      <th>Closing Count</th>
                      <th>Closing Total</th>
                      <th>Difference</th>
                    </tr>
                  </thead>
                  <tbody>
                    {noteValues.map(noteValue => {
                      const openingCount = reconciliation?.[`opening_${noteValue}_notes`] || 0;
                      const closingCount = reconciliation?.[`closing_${noteValue}_notes`] || 0;
                      const openingTotal = openingCount * noteValue;
                      const closingTotal = closingCount * noteValue;
                      const difference = closingTotal - openingTotal;

                      return (
                        <tr key={noteValue}>
                          <td className="note-value">
                            <strong>৳{noteValue}</strong>
                          </td>
                          <td>
                            <Form.Control
                              type="number"
                              min="0"
                              value={openingCount}
                              onChange={(e) => handleNoteChange(noteValue, 'opening', e.target.value)}
                              className="note-input"
                            />
                          </td>
                          <td className="total-cell">৳{openingTotal.toLocaleString()}</td>
                          <td>
                            <Form.Control
                              type="number"
                              min="0"
                              value={closingCount}
                              onChange={(e) => handleNoteChange(noteValue, 'closing', e.target.value)}
                              className="note-input"
                            />
                          </td>
                          <td className="total-cell">৳{closingTotal.toLocaleString()}</td>
                          <td className={`difference-cell ${difference === 0 ? 'text-success' : difference < 0 ? 'text-danger' : 'text-warning'}`}>
                            ৳{difference.toLocaleString()}
                          </td>
                        </tr>
                      );
                    })}
                  </tbody>
                <tfoot>
                  <tr className="total-row">
                    <th>TOTAL</th>
                    <th>-</th>
                    <th>৳{calculateTotal('opening').toLocaleString()}</th>
                    <th>-</th>
                    <th>৳{calculateTotal('closing').toLocaleString()}</th>
                    <th className={`${calculateVariance() === 0 ? 'text-success' : calculateVariance() < 0 ? 'text-danger' : 'text-warning'}`}>
                      ৳{(calculateTotal('closing') - calculateTotal('opening')).toLocaleString()}
                    </th>
                  </tr>
                </tfoot>
              </Table>
              </div>

              <div className="d-flex justify-content-between align-items-center mt-3">
                <div>
                  <small className="text-muted">
                    Last updated: {reconciliation?.updatedAt ? new Date(reconciliation.updatedAt).toLocaleString() : 'Never'}
                  </small>
                </div>
                <Button 
                  variant="primary" 
                  onClick={saveReconciliation}
                  disabled={saving}
                  className="save-btn"
                >
                  {saving ? (
                    <>
                      <Spinner size="sm" className="me-2" />
                      Saving...
                    </>
                  ) : (
                    <>
                      <i className="fas fa-save me-2"></i>
                      Save Reconciliation
                    </>
                  )}
                </Button>
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default CashBox;