import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Table, Form, Button, Alert, Spinner, Badge, Tab, Tabs } from 'react-bootstrap';
import axios from 'axios';
import './CashBoxReports.css';

const CashBoxReports = () => {
  const [activeTab, setActiveTab] = useState('daily');
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, message: '', type: '' });
  const [dailyData, setDailyData] = useState([]);
  const [weeklyData, setWeeklyData] = useState(null);
  const [monthlyData, setMonthlyData] = useState(null);
  const [dateRange, setDateRange] = useState({
    startDate: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
    endDate: new Date().toISOString().split('T')[0]
  });

  useEffect(() => {
    if (activeTab === 'daily') {
      fetchDailyReports();
    } else if (activeTab === 'weekly') {
      fetchWeeklyReport();
    } else if (activeTab === 'monthly') {
      fetchMonthlyReport();
    }
  }, [activeTab, dateRange]);

  const showAlert = (message, type) => {
    setAlert({ show: true, message, type });
    setTimeout(() => setAlert({ show: false, message: '', type: '' }), 5000);
  };

  const fetchDailyReports = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get(`http://localhost:5000/api/cash-reconciliation/range`, {
        headers: { Authorization: `Bearer ${token}` },
        params: {
          startDate: dateRange.startDate,
          endDate: dateRange.endDate
        }
      });
      setDailyData(response.data);
    } catch (error) {
      console.error('Error fetching daily reports:', error);
      showAlert('Error fetching daily reports', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchWeeklyReport = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get(`http://localhost:5000/api/cash-reconciliation/reports/weekly`, {
        headers: { Authorization: `Bearer ${token}` },
        params: {
          startDate: dateRange.startDate
        }
      });
      setWeeklyData(response.data);
    } catch (error) {
      console.error('Error fetching weekly report:', error);
      showAlert('Error fetching weekly report', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchMonthlyReport = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const currentDate = new Date();
      const response = await axios.get(`http://localhost:5000/api/cash-reconciliation/reports/monthly`, {
        headers: { Authorization: `Bearer ${token}` },
        params: {
          year: currentDate.getFullYear(),
          month: currentDate.getMonth() + 1
        }
      });
      setMonthlyData(response.data);
    } catch (error) {
      console.error('Error fetching monthly report:', error);
      showAlert('Error fetching monthly report', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const getVarianceBadge = (variance) => {
    if (variance === 0) return <Badge bg="success">Balanced</Badge>;
    if (variance < 0) return <Badge bg="danger">Short ৳{Math.abs(variance).toLocaleString()}</Badge>;
    return <Badge bg="warning">Over ৳{variance.toLocaleString()}</Badge>;
  };

  const exportToPDF = () => {
    showAlert('PDF export functionality will be implemented soon', 'info');
  };

  const exportToExcel = () => {
    showAlert('Excel export functionality will be implemented soon', 'info');
  };

  const renderDailyReports = () => (
    <Card className="reports-card">
      <Card.Header>
        <Row className="align-items-center">
          <Col>
            <h5 className="mb-0">
              <i className="fas fa-calendar-day me-2"></i>
              Daily Cash Reconciliation Reports
            </h5>
          </Col>
          <Col xs="auto">
            <div className="d-flex gap-2">
              <Button variant="outline-danger" size="sm" onClick={exportToPDF}>
                <i className="fas fa-file-pdf me-1"></i>PDF
              </Button>
              <Button variant="outline-success" size="sm" onClick={exportToExcel}>
                <i className="fas fa-file-excel me-1"></i>Excel
              </Button>
            </div>
          </Col>
        </Row>
      </Card.Header>
      <Card.Body>
        <Row className="mb-3">
          <Col md={4}>
            <Form.Group>
              <Form.Label>Start Date</Form.Label>
              <Form.Control
                type="date"
                value={dateRange.startDate}
                onChange={(e) => setDateRange(prev => ({ ...prev, startDate: e.target.value }))}
              />
            </Form.Group>
          </Col>
          <Col md={4}>
            <Form.Group>
              <Form.Label>End Date</Form.Label>
              <Form.Control
                type="date"
                value={dateRange.endDate}
                onChange={(e) => setDateRange(prev => ({ ...prev, endDate: e.target.value }))}
              />
            </Form.Group>
          </Col>
        </Row>

        {loading ? (
          <div className="text-center py-4">
            <Spinner animation="border" />
          </div>
        ) : (
          <Table responsive className="reports-table">
            <thead>
              <tr>
                <th>Date</th>
                <th>Opening Cash</th>
                <th>Closing Cash</th>
                <th>Total Income</th>
                <th>Total Expense</th>
                <th>Variance</th>
                <th>Status</th>
              </tr>
            </thead>
            <tbody>
              {dailyData.map(record => (
                <tr key={record.id}>
                  <td>{new Date(record.date).toLocaleDateString()}</td>
                  <td>৳{parseFloat(record.opening_total || 0).toLocaleString()}</td>
                  <td>৳{parseFloat(record.closing_total || 0).toLocaleString()}</td>
                  <td>৳{parseFloat(record.total_income || 0).toLocaleString()}</td>
                  <td>৳{parseFloat(record.total_expense || 0).toLocaleString()}</td>
                  <td className={`${record.variance === 0 ? 'text-success' : record.variance < 0 ? 'text-danger' : 'text-warning'}`}>
                    ৳{parseFloat(record.variance || 0).toLocaleString()}
                  </td>
                  <td>{getVarianceBadge(parseFloat(record.variance || 0))}</td>
                </tr>
              ))}
            </tbody>
          </Table>
        )}
      </Card.Body>
    </Card>
  );

  const renderWeeklyReport = () => (
    <Card className="reports-card">
      <Card.Header>
        <h5 className="mb-0">
          <i className="fas fa-calendar-week me-2"></i>
          Weekly Cash Reconciliation Summary
        </h5>
      </Card.Header>
      <Card.Body>
        {loading ? (
          <div className="text-center py-4">
            <Spinner animation="border" />
          </div>
        ) : weeklyData ? (
          <>
            <Row className="mb-4">
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Total Days</h6>
                    <h4>{weeklyData.summary.totalDays}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Avg Opening</h6>
                    <h4>৳{(weeklyData.summary.totalOpeningCash / weeklyData.summary.totalDays || 0).toLocaleString()}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Avg Closing</h6>
                    <h4>৳{(weeklyData.summary.totalClosingCash / weeklyData.summary.totalDays || 0).toLocaleString()}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Avg Variance</h6>
                    <h4 className={`${weeklyData.summary.averageVariance === 0 ? 'text-success' : weeklyData.summary.averageVariance < 0 ? 'text-danger' : 'text-warning'}`}>
                      ৳{weeklyData.summary.averageVariance.toLocaleString()}
                    </h4>
                  </Card.Body>
                </Card>
              </Col>
            </Row>

            <Table responsive className="reports-table">
              <thead>
                <tr>
                  <th>Date</th>
                  <th>Opening Cash</th>
                  <th>Closing Cash</th>
                  <th>Variance</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                {weeklyData.reconciliations.map(record => (
                  <tr key={record.id}>
                    <td>{new Date(record.date).toLocaleDateString()}</td>
                    <td>৳{parseFloat(record.opening_total || 0).toLocaleString()}</td>
                    <td>৳{parseFloat(record.closing_total || 0).toLocaleString()}</td>
                    <td className={`${record.variance === 0 ? 'text-success' : record.variance < 0 ? 'text-danger' : 'text-warning'}`}>
                      ৳{parseFloat(record.variance || 0).toLocaleString()}
                    </td>
                    <td>{getVarianceBadge(parseFloat(record.variance || 0))}</td>
                  </tr>
                ))}
              </tbody>
            </Table>
          </>
        ) : (
          <p className="text-muted text-center">No data available</p>
        )}
      </Card.Body>
    </Card>
  );

  const renderMonthlyReport = () => (
    <Card className="reports-card">
      <Card.Header>
        <h5 className="mb-0">
          <i className="fas fa-calendar-alt me-2"></i>
          Monthly Cash Reconciliation Summary
        </h5>
      </Card.Header>
      <Card.Body>
        {loading ? (
          <div className="text-center py-4">
            <Spinner animation="border" />
          </div>
        ) : monthlyData ? (
          <>
            <Row className="mb-4">
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Total Days</h6>
                    <h4>{monthlyData.summary.totalDays}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Total Opening</h6>
                    <h4>৳{monthlyData.summary.totalOpeningCash.toLocaleString()}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Total Closing</h6>
                    <h4>৳{monthlyData.summary.totalClosingCash.toLocaleString()}</h4>
                  </Card.Body>
                </Card>
              </Col>
              <Col md={3}>
                <Card className="summary-mini-card">
                  <Card.Body className="text-center">
                    <h6>Total Variance</h6>
                    <h4 className={`${monthlyData.summary.totalVariance === 0 ? 'text-success' : monthlyData.summary.totalVariance < 0 ? 'text-danger' : 'text-warning'}`}>
                      ৳{monthlyData.summary.totalVariance.toLocaleString()}
                    </h4>
                  </Card.Body>
                </Card>
              </Col>
            </Row>

            <Table responsive className="reports-table">
              <thead>
                <tr>
                  <th>Date</th>
                  <th>Opening Cash</th>
                  <th>Closing Cash</th>
                  <th>Variance</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                {monthlyData.reconciliations.map(record => (
                  <tr key={record.id}>
                    <td>{new Date(record.date).toLocaleDateString()}</td>
                    <td>৳{parseFloat(record.opening_total || 0).toLocaleString()}</td>
                    <td>৳{parseFloat(record.closing_total || 0).toLocaleString()}</td>
                    <td className={`${record.variance === 0 ? 'text-success' : record.variance < 0 ? 'text-danger' : 'text-warning'}`}>
                      ৳{parseFloat(record.variance || 0).toLocaleString()}
                    </td>
                    <td>{getVarianceBadge(parseFloat(record.variance || 0))}</td>
                  </tr>
                ))}
              </tbody>
            </Table>
          </>
        ) : (
          <p className="text-muted text-center">No data available</p>
        )}
      </Card.Body>
    </Card>
  );

  return (
    <Container fluid className="cash-box-reports-container">
      <Row className="mb-4">
        <Col>
          <h2 className="page-title">
            <i className="fas fa-chart-line me-2"></i>
            Cash Box Reports
          </h2>
          <p className="text-muted">View and analyze cash reconciliation reports</p>
        </Col>
      </Row>

      {alert.show && (
        <Alert variant={alert.type} dismissible onClose={() => setAlert({ show: false, message: '', type: '' })}>
          {alert.message}
        </Alert>
      )}

      <Tabs activeKey={activeTab} onSelect={(k) => setActiveTab(k)} className="mb-3">
        <Tab eventKey="daily" title={<span><i className="fas fa-calendar-day me-2"></i>Daily</span>}>
          {renderDailyReports()}
        </Tab>
        <Tab eventKey="weekly" title={<span><i className="fas fa-calendar-week me-2"></i>Weekly</span>}>
          {renderWeeklyReport()}
        </Tab>
        <Tab eventKey="monthly" title={<span><i className="fas fa-calendar-alt me-2"></i>Monthly</span>}>
          {renderMonthlyReport()}
        </Tab>
      </Tabs>
    </Container>
  );
};

export default CashBoxReports;