import React, { useState, useEffect } from 'react';
import { 
  Container, 
  Row, 
  Col, 
  Card, 
  Form, 
  Button, 
  Table, 
  Badge,
  Alert,
  Spinner
} from 'react-bootstrap';

const DSOReports = ({ showAlert }) => {
  const [filters, setFilters] = useState({
    startDate: new Date().toISOString().split('T')[0],
    endDate: new Date().toISOString().split('T')[0],
    dsoName: ''
  });

  const [reportData, setReportData] = useState([]);
  const [todaySummary, setTodaySummary] = useState({
    totalPaid: 0,
    totalCardsReceived: 0,
    totalCardsAllocated: 0,
    totalCardsSold: 0,
    simsSoldToday: false,
    kitsSoldToday: false
  });
  const [dsoNames, setDSONames] = useState([]);
  const [loading, setLoading] = useState(false);
  const [exporting, setExporting] = useState(false);

  useEffect(() => {
    fetchTodaySummary();
    fetchDSONames();
    fetchReportData();
  }, []);

  const fetchTodaySummary = async () => {
    try {
      const response = await fetch('/api/dso/today-summary');
      const data = await response.json();
      if (response.ok) {
        setTodaySummary(data);
      }
    } catch (error) {
      console.error('Error fetching today summary:', error);
    }
  };

  const fetchDSONames = async () => {
    try {
      const response = await fetch('/api/dso/dso-names');
      const data = await response.json();
      if (response.ok) {
        setDSONames(data || []);
      }
    } catch (error) {
      console.error('Error fetching DSO names:', error);
    }
  };

  const fetchReportData = async () => {
    setLoading(true);
    try {
      const queryParams = new URLSearchParams({
        startDate: filters.startDate,
        endDate: filters.endDate,
        ...(filters.dsoName && { dsoName: filters.dsoName })
      });

      const token = localStorage.getItem('token');
      const response = await fetch(`http://localhost:5000/api/dso/reports/daily?${queryParams}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      const data = await response.json();
      
      if (response.ok) {
        setReportData(data.report || []);
      } else {
        showAlert(data.message || 'Error fetching report data', 'danger');
      }
    } catch (error) {
      console.error('Error fetching report data:', error);
      showAlert('Error fetching report data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleFilterChange = (e) => {
    const { name, value } = e.target;
    setFilters(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleFilterSubmit = (e) => {
    e.preventDefault();
    fetchReportData();
  };

  const exportToCSV = async () => {
    setExporting(true);
    try {
      const queryParams = new URLSearchParams({
        startDate: filters.startDate,
        endDate: filters.endDate,
        ...(filters.dsoName && { dsoName: filters.dsoName }),
        format: 'csv'
      });

      const response = await fetch(`/api/dso/daily-report?${queryParams}`);
      
      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.style.display = 'none';
        a.href = url;
        a.download = `dso-activity-report-${filters.startDate}-to-${filters.endDate}.csv`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        showAlert('CSV report exported successfully!', 'success');
      } else {
        showAlert('Error exporting CSV report', 'danger');
      }
    } catch (error) {
      console.error('Error exporting CSV:', error);
      showAlert('Error exporting CSV report', 'danger');
    } finally {
      setExporting(false);
    }
  };

  const exportToExcel = async () => {
    setExporting(true);
    try {
      const queryParams = new URLSearchParams({
        startDate: filters.startDate,
        endDate: filters.endDate,
        ...(filters.dsoName && { dsoName: filters.dsoName }),
        format: 'excel'
      });

      const response = await fetch(`/api/dso/daily-report?${queryParams}`);
      
      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.style.display = 'none';
        a.href = url;
        a.download = `dso-activity-report-${filters.startDate}-to-${filters.endDate}.xlsx`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        showAlert('Excel report exported successfully!', 'success');
      } else {
        showAlert('Error exporting Excel report', 'danger');
      }
    } catch (error) {
      console.error('Error exporting Excel:', error);
      showAlert('Error exporting Excel report', 'danger');
    } finally {
      setExporting(false);
    }
  };

  const formatCardTypes = (cardTypes) => {
    if (!cardTypes || cardTypes.length === 0) return '-';
    return cardTypes.map(ct => `${ct.type}: ${ct.quantity}`).join(', ');
  };

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h3 className="text-primary">
            <i className="fas fa-chart-line me-2"></i>
            DSO Activity Reports
          </h3>
          <p className="text-muted">Daily reports and analytics for DSO deliveries, allocations, and sales</p>
        </Col>
      </Row>

      {/* Today's Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="bg-primary text-white">
            <Card.Body>
              <div className="d-flex justify-content-between">
                <div>
                  <h6 className="card-title">Total Paid Today</h6>
                  <h4>৳{parseFloat(todaySummary.totalPaid).toFixed(2)}</h4>
                </div>
                <div className="align-self-center">
                  <i className="fas fa-money-bill-wave fa-2x"></i>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="bg-success text-white">
            <Card.Body>
              <div className="d-flex justify-content-between">
                <div>
                  <h6 className="card-title">Cards Received</h6>
                  <h4>{todaySummary.totalCardsReceived}</h4>
                </div>
                <div className="align-self-center">
                  <i className="fas fa-credit-card fa-2x"></i>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="bg-info text-white">
            <Card.Body>
              <div className="d-flex justify-content-between">
                <div>
                  <h6 className="card-title">Cards Allocated</h6>
                  <h4>{todaySummary.totalCardsAllocated}</h4>
                </div>
                <div className="align-self-center">
                  <i className="fas fa-share-alt fa-2x"></i>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="bg-warning text-white">
            <Card.Body>
              <div className="d-flex justify-content-between">
                <div>
                  <h6 className="card-title">Cards Sold Today</h6>
                  <h4>{todaySummary.totalCardsSold}</h4>
                </div>
                <div className="align-self-center">
                  <i className="fas fa-shopping-cart fa-2x"></i>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Additional Summary Info */}
      <Row className="mb-4">
        <Col md={6}>
          <Alert variant="info">
            <strong>SIMs Sold Today:</strong> {todaySummary.simsSoldToday ? 'Yes' : 'No'}
          </Alert>
        </Col>
        <Col md={6}>
          <Alert variant="info">
            <strong>Kits Sold Today:</strong> {todaySummary.kitsSoldToday ? 'Yes' : 'No'}
          </Alert>
        </Col>
      </Row>

      {/* Filters */}
      <Card className="mb-4">
        <Card.Header>
          <h5 className="mb-0">Report Filters</h5>
        </Card.Header>
        <Card.Body>
          <Form onSubmit={handleFilterSubmit}>
            <Row>
              <Col md={3}>
                <Form.Group className="mb-3">
                  <Form.Label>Start Date</Form.Label>
                  <Form.Control
                    type="date"
                    name="startDate"
                    value={filters.startDate}
                    onChange={handleFilterChange}
                  />
                </Form.Group>
              </Col>
              <Col md={3}>
                <Form.Group className="mb-3">
                  <Form.Label>End Date</Form.Label>
                  <Form.Control
                    type="date"
                    name="endDate"
                    value={filters.endDate}
                    onChange={handleFilterChange}
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>DSO Name (Optional)</Form.Label>
                  <Form.Control
                    type="text"
                    name="dsoName"
                    value={filters.dsoName}
                    onChange={handleFilterChange}
                    list="dso-names-filter"
                    placeholder="Filter by DSO name..."
                  />
                  <datalist id="dso-names-filter">
                    {dsoNames.map((name, index) => (
                      <option key={index} value={name} />
                    ))}
                  </datalist>
                </Form.Group>
              </Col>
              <Col md={2}>
                <Form.Group className="mb-3">
                  <Form.Label>&nbsp;</Form.Label>
                  <div>
                    <Button type="submit" variant="primary" disabled={loading}>
                      {loading ? <Spinner size="sm" /> : <i className="fas fa-search"></i>}
                      {loading ? ' Loading...' : ' Filter'}
                    </Button>
                  </div>
                </Form.Group>
              </Col>
            </Row>
          </Form>
        </Card.Body>
      </Card>

      {/* Export Buttons */}
      <Row className="mb-3">
        <Col>
          <div className="d-flex gap-2">
            <Button
              variant="outline-success"
              onClick={exportToCSV}
              disabled={exporting || reportData.length === 0}
            >
              {exporting ? <Spinner size="sm" /> : <i className="fas fa-file-csv"></i>}
              {exporting ? ' Exporting...' : ' Export CSV'}
            </Button>
            <Button
              variant="outline-primary"
              onClick={exportToExcel}
              disabled={exporting || reportData.length === 0}
            >
              {exporting ? <Spinner size="sm" /> : <i className="fas fa-file-excel"></i>}
              {exporting ? ' Exporting...' : ' Export Excel'}
            </Button>
          </div>
        </Col>
      </Row>

      {/* Report Table */}
      <Card>
        <Card.Header>
          <h5 className="mb-0">DSO Activity — Daily Report</h5>
        </Card.Header>
        <Card.Body>
          {loading ? (
            <div className="text-center py-4">
              <Spinner animation="border" role="status">
                <span className="visually-hidden">Loading...</span>
              </Spinner>
              <p className="mt-2">Loading report data...</p>
            </div>
          ) : reportData.length === 0 ? (
            <Alert variant="info">
              <i className="fas fa-info-circle me-2"></i>
              No data found for the selected date range and filters.
            </Alert>
          ) : (
            <div className="table-responsive">
              <Table striped bordered hover>
                <thead className="table-dark">
                  <tr>
                    <th>Date</th>
                    <th>DSO Name</th>
                    <th>Amount Paid (৳)</th>
                    <th>Cards Received</th>
                    <th>Card Types & Qtys</th>
                    <th>SIMs Received</th>
                    <th>Kits Received</th>
                    <th>Cards Allocated</th>
                    <th>SIMs Allocated</th>
                    <th>Kits Allocated</th>
                    <th>Cards Sold</th>
                    <th>SIMs Sold</th>
                    <th>Kits Sold</th>
                    <th>Received By</th>
                    <th>Notes</th>
                  </tr>
                </thead>
                <tbody>
                  {reportData.map((row, index) => (
                    <tr key={index}>
                      <td>{row.date}</td>
                      <td>{row.dsoName}</td>
                      <td>৳{parseFloat(row.amountPaid || 0).toFixed(2)}</td>
                      <td>{row.cardsReceived || 0}</td>
                      <td className="small">{formatCardTypes(row.cardTypes)}</td>
                      <td>{row.simsReceived || 0}</td>
                      <td>{row.kitsReceived || 0}</td>
                      <td>
                        {row.cardsAllocated || 0}
                        {row.cardsAllocated > 0 && (
                          <Badge bg="info" className="ms-1">A</Badge>
                        )}
                      </td>
                      <td>
                        {row.simsAllocated || 0}
                        {row.simsAllocated > 0 && (
                          <Badge bg="info" className="ms-1">A</Badge>
                        )}
                      </td>
                      <td>
                        {row.kitsAllocated || 0}
                        {row.kitsAllocated > 0 && (
                          <Badge bg="info" className="ms-1">A</Badge>
                        )}
                      </td>
                      <td>
                        {row.cardsSold || 0}
                        {row.cardsSold > 0 && (
                          <Badge bg="success" className="ms-1">S</Badge>
                        )}
                      </td>
                      <td>
                        {row.simsSold || 0}
                        {row.simsSold > 0 && (
                          <Badge bg="success" className="ms-1">S</Badge>
                        )}
                      </td>
                      <td>
                        {row.kitsSold || 0}
                        {row.kitsSold > 0 && (
                          <Badge bg="success" className="ms-1">S</Badge>
                        )}
                      </td>
                      <td>{row.receivedBy}</td>
                      <td className="small">{row.notes || '-'}</td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </div>
          )}
        </Card.Body>
      </Card>

      {/* Legend */}
      <Row className="mt-3">
        <Col>
          <Card className="bg-light">
            <Card.Body className="py-2">
              <small>
                <strong>Legend:</strong>
                <Badge bg="info" className="ms-2 me-1">A</Badge> Allocated
                <Badge bg="success" className="ms-2 me-1">S</Badge> Sold
                <span className="ms-3"><strong>Card Types Format:</strong> Type: Quantity, Type: Quantity</span>
              </small>
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default DSOReports;