import React, { useState, useEffect } from 'react';
import { Row, Col, Card, Form, Button, Table, Badge, Spinner, Modal, Tab, Tabs } from 'react-bootstrap';
import axios from 'axios';

const MobileRechargeOperator = ({ operator, operatorName, operatorColor, showAlert }) => {
  const [accountData, setAccountData] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('account');
  const [formData, setFormData] = useState({
    openingBalance: '',
    closingBalance: '',
    commissionRate: '2.80',
    date: new Date().toISOString().split('T')[0]
  });
  const [transactionForm, setTransactionForm] = useState({
    customerNumber: '',
    rechargeAmount: '',
    commission: '',
    srPay: '',
    minutesCard: '',
    mbCard: '',
    simQuantity: '',
    transactionType: 'recharge',
    reference: '',
    notes: '',
    date: new Date().toISOString().split('T')[0],
    time: new Date().toTimeString().split(' ')[0].substring(0, 5)
  });
  const [showTransactionModal, setShowTransactionModal] = useState(false);
  const [showOpenDayModal, setShowOpenDayModal] = useState(false);
  const [showCloseDayModal, setShowCloseDayModal] = useState(false);
  const [summary, setSummary] = useState(null);
  const [dailySummary, setDailySummary] = useState(null);
  const [openDayForm, setOpenDayForm] = useState({
    openingBalance: '',
    notes: ''
  });
  const [closeDayForm, setCloseDayForm] = useState({
    closingBalance: '',
    notes: ''
  });
  
  // SMS Testing state
  const [smsForm, setSmsForm] = useState({
    smsText: '',
    senderNumber: '',
    messageId: ''
  });
  const [smsResult, setSmsResult] = useState(null);
  const [smsLoading, setSmsLoading] = useState(false);

  useEffect(() => {
    fetchTodayAccount();
    fetchTodayTransactions();
    fetchDailySummary();
  }, [operator]);

  const fetchTodayAccount = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get(`http://localhost:5000/api/mobile-recharge/${operator}/today`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAccountData(response.data);
      if (response.data) {
        setFormData({
          openingBalance: response.data.openingBalance || '',
          closingBalance: response.data.closingBalance || '',
          commissionRate: response.data.commissionRate || '2.80',
          date: response.data.date || new Date().toISOString().split('T')[0]
        });
      }
    } catch (error) {
      console.error('Error fetching today account:', error);
      if (error.response?.status !== 404) {
        showAlert('Error fetching account data', 'danger');
      }
    } finally {
      setLoading(false);
    }
  };

  const fetchTodayTransactions = async () => {
    try {
      const token = localStorage.getItem('token');
      const today = new Date().toISOString().split('T')[0];
      const response = await axios.get(`http://localhost:5000/api/mobile-recharge/${operator}/transactions?date=${today}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTransactions(response.data);
    } catch (error) {
      console.error('Error fetching transactions:', error);
    }
  };

  const fetchDailySummary = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get(`http://localhost:5000/api/mobile-recharge/${operator}/daily-summary`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setDailySummary(response.data);
    } catch (error) {
      console.error('Error fetching daily summary:', error);
    }
  };

  const handleOpenDay = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        openingBalance: parseFloat(openDayForm.openingBalance) || 0,
        notes: openDayForm.notes
      };

      await axios.post(`http://localhost:5000/api/mobile-recharge/${operator}/open-day`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });

      showAlert(`Day opened successfully for ${operatorName}!`, 'success');
      setShowOpenDayModal(false);
      setOpenDayForm({ openingBalance: '', notes: '' });
      fetchTodayAccount();
      fetchDailySummary();
    } catch (error) {
      console.error('Error opening day:', error);
      showAlert(error.response?.data?.message || 'Error opening day', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleCloseDay = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        closingBalance: parseFloat(closeDayForm.closingBalance) || 0,
        notes: closeDayForm.notes
      };

      await axios.post(`http://localhost:5000/api/mobile-recharge/${operator}/close-day`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });

      showAlert(`Day closed successfully for ${operatorName}!`, 'success');
      setShowCloseDayModal(false);
      setCloseDayForm({ closingBalance: '', notes: '' });
      fetchTodayAccount();
      fetchDailySummary();
    } catch (error) {
      console.error('Error closing day:', error);
      showAlert(error.response?.data?.message || 'Error closing day', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleTransactionInputChange = (e) => {
    const { name, value } = e.target;
    setTransactionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  // SMS handling functions
  const handleSmsInputChange = (e) => {
    const { name, value } = e.target;
    setSmsForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleTestSMSParsing = async (e) => {
    e.preventDefault();
    try {
      setSmsLoading(true);
      setSmsResult(null);

      const response = await axios.post('http://localhost:5000/api/sms/test', {
        smsText: smsForm.smsText
      });

      setSmsResult({
        type: 'test',
        success: true,
        data: response.data
      });

    } catch (error) {
      console.error('Error testing SMS parsing:', error);
      setSmsResult({
        type: 'test',
        success: false,
        error: error.response?.data?.message || 'Error testing SMS parsing'
      });
    } finally {
      setSmsLoading(false);
    }
  };

  const handleProcessSMS = async (e) => {
    e.preventDefault();
    try {
      setSmsLoading(true);
      setSmsResult(null);

      const response = await axios.post('http://localhost:5000/api/sms/receive', {
        smsText: smsForm.smsText,
        senderNumber: smsForm.senderNumber,
        messageId: smsForm.messageId || `TEST-${Date.now()}`,
        receivedAt: new Date().toISOString()
      });

      setSmsResult({
        type: 'process',
        success: true,
        data: response.data
      });

      // Refresh the data
      fetchTodayAccount();
      fetchTodayTransactions();
      fetchDailySummary();

      showAlert('SMS processed successfully and recharge transaction created!', 'success');

    } catch (error) {
      console.error('Error processing SMS:', error);
      setSmsResult({
        type: 'process',
        success: false,
        error: error.response?.data?.message || 'Error processing SMS'
      });
      showAlert(error.response?.data?.message || 'Error processing SMS', 'danger');
    } finally {
      setSmsLoading(false);
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...formData
      };

      // Only include numeric values if they are not empty
      if (formData.openingBalance !== '') {
        submitData.openingBalance = parseFloat(formData.openingBalance);
      }
      if (formData.closingBalance !== '') {
        submitData.closingBalance = parseFloat(formData.closingBalance);
      }

      await axios.put(`http://localhost:5000/api/mobile-recharge/${operator}`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });

      showAlert(`${operatorName} account updated successfully!`, 'success');
      fetchTodayAccount();
    } catch (error) {
      console.error('Error updating account:', error);
      showAlert(error.response?.data?.message || 'Error updating account', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleAddTransaction = async (e) => {
    e.preventDefault();
    try {
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...transactionForm,
        rechargeAmount: parseFloat(transactionForm.rechargeAmount) || 0,
        commission: parseFloat(transactionForm.commission) || 0
      };

      await axios.post(`http://localhost:5000/api/mobile-recharge/${operator}/transaction`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });

      showAlert('Transaction added successfully!', 'success');
      setShowTransactionModal(false);
      setTransactionForm({
        customerNumber: '',
        rechargeAmount: '',
        commission: '',
        srPay: '',
        minutesCard: '',
        mbCard: '',
        simQuantity: '',
        transactionType: 'recharge',
        reference: '',
        notes: '',
        date: new Date().toISOString().split('T')[0],
        time: new Date().toTimeString().split(' ')[0].substring(0, 5)
      });
      fetchTodayTransactions();
      fetchTodayAccount();
    } catch (error) {
      console.error('Error adding transaction:', error);
      showAlert('Error adding transaction', 'danger');
    }
  };

  const handleDeleteTransaction = async (transactionId) => {
    if (window.confirm('Are you sure you want to delete this transaction?')) {
      try {
        const token = localStorage.getItem('token');
        await axios.delete(`http://localhost:5000/api/mobile-recharge/transaction/${transactionId}`, {
          headers: { Authorization: `Bearer ${token}` }
        });
        showAlert('Transaction deleted successfully!', 'success');
        fetchTodayTransactions();
        fetchTodayAccount();
      } catch (error) {
        console.error('Error deleting transaction:', error);
        showAlert('Error deleting transaction', 'danger');
      }
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-BD', {
      style: 'currency',
      currency: 'BDT',
      minimumFractionDigits: 2
    }).format(amount || 0);
  };

  const getTransactionBadgeClass = (type) => {
    switch (type) {
      case 'recharge': return 'bg-success';
      case 'commission': return 'bg-info';
      case 'adjustment': return 'bg-warning';
      default: return 'bg-secondary';
    }
  };

  return (
    <div className="operator-content">
      <Row className="mb-4">
        <Col>
          <Card className="operator-card">
            <Card.Header className="operator-header" style={{ background: `linear-gradient(135deg, ${operatorColor}, ${operatorColor}dd)` }}>
              <h4 className="mb-0">
                <i className="fas fa-mobile-alt me-2"></i>
                {operatorName} Account Management
              </h4>
            </Card.Header>
            <Card.Body>
              {/* Daily Balance Management */}
              <Row className="mb-3">
                <Col>
                  <div className="d-flex justify-content-between align-items-center mb-3">
                    <h6 className="mb-0">Daily Balance Management</h6>
                    <div>
                      {dailySummary && !dailySummary.isOpened && (
                        <Button 
                          variant="success" 
                          size="sm" 
                          className="me-2"
                          onClick={() => setShowOpenDayModal(true)}
                        >
                          <i className="fas fa-play me-1"></i>
                          Open Day
                        </Button>
                      )}
                      {dailySummary && dailySummary.isOpened && !dailySummary.isClosed && (
                        <Button 
                          variant="danger" 
                          size="sm"
                          onClick={() => setShowCloseDayModal(true)}
                        >
                          <i className="fas fa-stop me-1"></i>
                          Close Day
                        </Button>
                      )}
                    </div>
                  </div>
                  
                  {dailySummary && (
                    <Row className="g-2">
                      <Col md={3}>
                        <div className="text-center p-2 border rounded">
                          <small className="text-muted d-block">Status</small>
                          <span className={`badge ${dailySummary.isClosed ? 'bg-danger' : dailySummary.isOpened ? 'bg-success' : 'bg-warning'}`}>
                            {dailySummary.isClosed ? 'Closed' : dailySummary.isOpened ? 'Open' : 'Not Opened'}
                          </span>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="text-center p-2 border rounded">
                          <small className="text-muted d-block">Opening Balance</small>
                          <strong className="text-primary">
                            {dailySummary.openingBalance !== null ? formatCurrency(dailySummary.openingBalance) : 'Not Set'}
                          </strong>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="text-center p-2 border rounded">
                          <small className="text-muted d-block">Closing Balance</small>
                          <strong className="text-danger">
                            {dailySummary.closingBalance !== null ? formatCurrency(dailySummary.closingBalance) : 'Not Set'}
                          </strong>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="text-center p-2 border rounded">
                          <small className="text-muted d-block">Net Change</small>
                          <strong className={dailySummary.netChange > 0 ? 'text-success' : dailySummary.netChange < 0 ? 'text-danger' : 'text-muted'}>
                            {dailySummary.netChange !== null ? formatCurrency(dailySummary.netChange) : 'N/A'}
                          </strong>
                        </div>
                      </Col>
                    </Row>
                  )}
                </Col>
              </Row>
              <hr />
              <Tabs activeKey={activeTab} onSelect={(k) => setActiveTab(k)} className="mb-3">
                <Tab eventKey="account" title={<span><i className="fas fa-wallet me-2"></i>Account</span>}>
                  <p className="text-muted mb-3">
                    <small>Set opening balance at the start of the day and closing balance at the end. When you enter closing balance, total recharge and commission will be calculated automatically based on the commission rate.</small>
                  </p>
                  <Form onSubmit={handleSubmit}>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Date</Form.Label>
                          <Form.Control
                            type="date"
                            name="date"
                            value={formData.date}
                            onChange={handleInputChange}
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Commission Rate (%)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="commissionRate"
                            value={formData.commissionRate}
                            onChange={handleInputChange}
                            placeholder="Enter commission rate (e.g., 2.80)"
                          />
                          <Form.Text className="text-muted">
                            Commission percentage (e.g., 2.80 for 2.8%)
                          </Form.Text>
                        </Form.Group>
                      </Col>
                    </Row>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Opening Balance</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="openingBalance"
                            value={formData.openingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter opening balance"
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Closing Balance <small className="text-muted">(Optional - Auto calculates recharge & commission)</small></Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="closingBalance"
                            value={formData.closingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter closing balance (optional)"
                          />
                          <Form.Text className="text-muted">
                            When entered, total recharge = opening balance - closing balance
                          </Form.Text>
                        </Form.Group>
                      </Col>
                    </Row>
                    <Button 
                      type="submit" 
                      className="btn-operator"
                      disabled={loading}
                      style={{ backgroundColor: operatorColor, borderColor: operatorColor }}
                    >
                      {loading ? <Spinner size="sm" className="me-2" /> : null}
                      Update Account
                    </Button>
                  </Form>

                  {accountData && (
                    <Row className="mt-4">
                      <Col>
                        <Card className="balance-card" style={{ background: `linear-gradient(135deg, ${operatorColor}, ${operatorColor}dd)` }}>
                          <Card.Body>
                            <Row>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.openingBalance)}</div>
                                <div className="balance-label">Opening Balance</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.totalRecharge)}</div>
                                <div className="balance-label">Total Recharge</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.totalCommission)}</div>
                                <div className="balance-label">Total Commission</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{accountData.commissionRate || 2.80}%</div>
                                <div className="balance-label">Commission Rate</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.totalSrPay || 0)}</div>
                                <div className="balance-label">Total SR Pay</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.totalMinutesCard || 0)}</div>
                                <div className="balance-label">Total Minutes Card</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.totalMbCard || 0)}</div>
                                <div className="balance-label">Total MB Card</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{accountData.totalSimQuantity || 0}</div>
                                <div className="balance-label">Total SIM Quantity</div>
                              </Col>
                              <Col lg={2} md={4} sm={6} className="balance-item">
                                <div className="balance-amount">{formatCurrency(accountData.closingBalance)}</div>
                                <div className="balance-label">Closing Balance</div>
                              </Col>
                            </Row>
                          </Card.Body>
                        </Card>
                      </Col>
                    </Row>
                  )}
                </Tab>

                <Tab eventKey="transactions" title={<span><i className="fas fa-list me-2"></i>Transactions</span>}>
                  <div className="d-flex justify-content-between align-items-center mb-3">
                    <h5>Today's Transactions</h5>
                    <Button 
                      variant="primary" 
                      onClick={() => setShowTransactionModal(true)}
                      style={{ backgroundColor: operatorColor, borderColor: operatorColor }}
                    >
                      <i className="fas fa-plus me-2"></i>Add Transaction
                    </Button>
                  </div>

                  <Table responsive className="transaction-table">
                    <thead>
                      <tr>
                        <th>Time</th>
                        <th>Customer Number</th>
                        <th>Type</th>
                        <th>Recharge Amount</th>
                        <th>Commission</th>
                        <th>SR Pay</th>
                        <th>Minutes Card</th>
                        <th>MB Card</th>
                        <th>SIM Qty</th>
                        <th>Reference</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      {transactions.map((transaction) => (
                        <tr key={transaction.id}>
                          <td>{transaction.time}</td>
                          <td>{transaction.customerNumber}</td>
                          <td>
                            <Badge className={getTransactionBadgeClass(transaction.transactionType)}>
                              {transaction.transactionType}
                            </Badge>
                          </td>
                          <td>{formatCurrency(transaction.rechargeAmount)}</td>
                          <td>{formatCurrency(transaction.commission)}</td>
                          <td>{formatCurrency(transaction.srPay || 0)}</td>
                          <td>{formatCurrency(transaction.minutesCard || 0)}</td>
                          <td>{formatCurrency(transaction.mbCard || 0)}</td>
                          <td>{transaction.simQuantity || 0}</td>
                          <td>{transaction.reference}</td>
                          <td>
                            <Button
                              variant="outline-danger"
                              size="sm"
                              onClick={() => handleDeleteTransaction(transaction.id)}
                            >
                              <i className="fas fa-trash"></i>
                            </Button>
                          </td>
                        </tr>
                      ))}
                      {transactions.length === 0 && (
                        <tr>
                          <td colSpan="7" className="text-center text-muted">
                            No transactions found for today
                          </td>
                        </tr>
                      )}
                    </tbody>
                  </Table>
                </Tab>

                <Tab eventKey="sms" title={<span><i className="fas fa-sms me-2"></i>SMS Testing</span>}>
                  <div className="mb-3">
                    <h5>SMS-Based Automatic Recharge Testing</h5>
                    <p className="text-muted">
                      Test the SMS parsing and automatic recharge functionality. Enter an SMS text that contains recharge information.
                    </p>
                  </div>

                  <Form onSubmit={handleTestSMSParsing}>
                    <Row>
                      <Col md={12}>
                        <Form.Group className="mb-3">
                          <Form.Label>SMS Text</Form.Label>
                          <Form.Control
                            as="textarea"
                            rows={3}
                            name="smsText"
                            value={smsForm.smsText}
                            onChange={handleSmsInputChange}
                            placeholder="Enter SMS text (e.g., 'Grameenphone recharge of Tk 100 to 01712345678')"
                            required
                          />
                          <Form.Text className="text-muted">
                            Example formats: "GP recharge Tk 100 to 01712345678", "01712345678 recharged with Tk 50 from Banglalink", "Tk 200 recharge to 01812345678 Robi"
                          </Form.Text>
                        </Form.Group>
                      </Col>
                    </Row>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Sender Number (Optional)</Form.Label>
                          <Form.Control
                            type="text"
                            name="senderNumber"
                            value={smsForm.senderNumber}
                            onChange={handleSmsInputChange}
                            placeholder="e.g., 16247"
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Message ID (Optional)</Form.Label>
                          <Form.Control
                            type="text"
                            name="messageId"
                            value={smsForm.messageId}
                            onChange={handleSmsInputChange}
                            placeholder="e.g., MSG123456"
                          />
                        </Form.Group>
                      </Col>
                    </Row>
                    <div className="d-flex gap-2 mb-3">
                      <Button 
                        type="submit" 
                        variant="outline-primary"
                        disabled={smsLoading || !smsForm.smsText}
                      >
                        {smsLoading ? <Spinner size="sm" className="me-2" /> : <i className="fas fa-search me-2"></i>}
                        Test SMS Parsing
                      </Button>
                      <Button 
                        variant="primary"
                        onClick={handleProcessSMS}
                        disabled={smsLoading || !smsForm.smsText}
                        style={{ backgroundColor: operatorColor, borderColor: operatorColor }}
                      >
                        {smsLoading ? <Spinner size="sm" className="me-2" /> : <i className="fas fa-play me-2"></i>}
                        Process SMS & Create Transaction
                      </Button>
                    </div>
                  </Form>

                  {smsResult && (
                    <Card className={`mt-3 ${smsResult.success ? 'border-success' : 'border-danger'}`}>
                      <Card.Header className={`${smsResult.success ? 'bg-success text-white' : 'bg-danger text-white'}`}>
                        <h6 className="mb-0">
                          <i className={`fas ${smsResult.success ? 'fa-check-circle' : 'fa-exclamation-triangle'} me-2`}></i>
                          {smsResult.type === 'test' ? 'SMS Parsing Result' : 'SMS Processing Result'}
                        </h6>
                      </Card.Header>
                      <Card.Body>
                        {smsResult.success ? (
                          <div>
                            {smsResult.type === 'test' && smsResult.data.parsedData ? (
                              <div>
                                <h6>Parsed Information:</h6>
                                <Row>
                                  <Col md={6}>
                                    <strong>Operator:</strong> {smsResult.data.parsedData.operator}<br/>
                                    <strong>Phone Number:</strong> {smsResult.data.parsedData.phoneNumber}<br/>
                                    <strong>Amount:</strong> ৳{smsResult.data.parsedData.amount}
                                  </Col>
                                  <Col md={6}>
                                    <strong>Commission Rate:</strong> {smsResult.data.parsedData.commissionRate}%<br/>
                                    <strong>Commission:</strong> ৳{smsResult.data.parsedData.commission.toFixed(2)}<br/>
                                    <strong>Can Process:</strong> <Badge bg="success">Yes</Badge>
                                  </Col>
                                </Row>
                              </div>
                            ) : smsResult.type === 'test' ? (
                              <div className="text-warning">
                                <i className="fas fa-exclamation-triangle me-2"></i>
                                Could not parse recharge information from this SMS text.
                              </div>
                            ) : (
                              <div>
                                <h6>Transaction Created Successfully!</h6>
                                <Row>
                                  <Col md={6}>
                                    <strong>Operator:</strong> {smsResult.data.data.parsedData.operator}<br/>
                                    <strong>Phone Number:</strong> {smsResult.data.data.parsedData.phoneNumber}<br/>
                                    <strong>Amount:</strong> ৳{smsResult.data.data.parsedData.amount}
                                  </Col>
                                  <Col md={6}>
                                    <strong>Commission:</strong> ৳{smsResult.data.data.parsedData.commission.toFixed(2)}<br/>
                                    <strong>Account Updated:</strong> <Badge bg="success">Yes</Badge><br/>
                                    <strong>Transaction ID:</strong> {smsResult.data.data.transaction.id}
                                  </Col>
                                </Row>
                              </div>
                            )}
                          </div>
                        ) : (
                          <div className="text-danger">
                            <i className="fas fa-exclamation-triangle me-2"></i>
                            {smsResult.error}
                          </div>
                        )}
                      </Card.Body>
                    </Card>
                  )}

                  <Card className="mt-4">
                    <Card.Header>
                      <h6 className="mb-0">
                        <i className="fas fa-info-circle me-2"></i>
                        SMS Format Examples
                      </h6>
                    </Card.Header>
                    <Card.Body>
                      <div className="row">
                        <div className="col-md-4">
                          <h6>Grameenphone:</h6>
                          <small className="text-muted">
                            • "GP recharge Tk 100 to 01712345678"<br/>
                            • "Grameenphone recharge of Tk 50 to 01712345678"<br/>
                            • "01712345678 recharged with Tk 200 from GP"
                          </small>
                        </div>
                        <div className="col-md-4">
                          <h6>Banglalink:</h6>
                          <small className="text-muted">
                            • "BL recharge Tk 100 to 01912345678"<br/>
                            • "Banglalink recharge of Tk 50 to 01912345678"<br/>
                            • "01912345678 recharged with Tk 200 from Banglalink"
                          </small>
                        </div>
                        <div className="col-md-4">
                          <h6>Robi/Airtel:</h6>
                          <small className="text-muted">
                            • "Robi recharge Tk 100 to 01812345678"<br/>
                            • "Airtel recharge of Tk 50 to 01812345678"<br/>
                            • "01812345678 recharged with Tk 200 from Robi"
                          </small>
                        </div>
                      </div>
                    </Card.Body>
                  </Card>
                </Tab>
              </Tabs>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Add Transaction Modal */}
      <Modal show={showTransactionModal} onHide={() => setShowTransactionModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>Add {operatorName} Transaction</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form onSubmit={handleAddTransaction}>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Date</Form.Label>
                  <Form.Control
                    type="date"
                    name="date"
                    value={transactionForm.date}
                    onChange={handleTransactionInputChange}
                    required
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Time</Form.Label>
                  <Form.Control
                    type="time"
                    name="time"
                    value={transactionForm.time}
                    onChange={handleTransactionInputChange}
                    required
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Customer Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="customerNumber"
                    value={transactionForm.customerNumber}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter customer mobile number"
                    required
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Transaction Type</Form.Label>
                  <Form.Select
                    name="transactionType"
                    value={transactionForm.transactionType}
                    onChange={handleTransactionInputChange}
                    required
                  >
                    <option value="recharge">Recharge</option>
                    <option value="commission">Commission</option>
                    <option value="adjustment">Adjustment</option>
                  </Form.Select>
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>Recharge Amount</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="rechargeAmount"
                    value={transactionForm.rechargeAmount}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter recharge amount"
                    required
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>Commission</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="commission"
                    value={transactionForm.commission}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter commission amount"
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>SR Pay</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="srPay"
                    value={transactionForm.srPay}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter SR pay amount"
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>Minutes Card</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="minutesCard"
                    value={transactionForm.minutesCard}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter minutes card amount"
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>MB Card</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.01"
                    name="mbCard"
                    value={transactionForm.mbCard}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter MB card amount"
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>SIM Quantity</Form.Label>
                  <Form.Control
                    type="number"
                    name="simQuantity"
                    value={transactionForm.simQuantity}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter SIM quantity"
                    min="0"
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Reference</Form.Label>
                  <Form.Control
                    type="text"
                    name="reference"
                    value={transactionForm.reference}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter reference number"
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Notes</Form.Label>
                  <Form.Control
                    type="text"
                    name="notes"
                    value={transactionForm.notes}
                    onChange={handleTransactionInputChange}
                    placeholder="Enter notes (optional)"
                  />
                </Form.Group>
              </Col>
            </Row>
            <div className="d-flex justify-content-end">
              <Button variant="secondary" className="me-2" onClick={() => setShowTransactionModal(false)}>
                Cancel
              </Button>
              <Button 
                type="submit" 
                style={{ backgroundColor: operatorColor, borderColor: operatorColor }}
              >
                Add Transaction
              </Button>
            </div>
          </Form>
        </Modal.Body>
      </Modal>

      {/* Open Day Modal */}
      <Modal show={showOpenDayModal} onHide={() => setShowOpenDayModal(false)} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-play me-2" style={{ color: operatorColor }}></i>
            Open Day - {operatorName}
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form onSubmit={handleOpenDay}>
            <Form.Group className="mb-3">
              <Form.Label>Opening Balance <span className="text-danger">*</span></Form.Label>
              <Form.Control
                type="number"
                step="0.01"
                name="openingBalance"
                value={openDayForm.openingBalance}
                onChange={(e) => setOpenDayForm({...openDayForm, openingBalance: e.target.value})}
                placeholder="Enter opening balance"
                required
              />
              <Form.Text className="text-muted">
                Enter the starting balance for today's operations
              </Form.Text>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Notes</Form.Label>
              <Form.Control
                as="textarea"
                rows={3}
                name="notes"
                value={openDayForm.notes}
                onChange={(e) => setOpenDayForm({...openDayForm, notes: e.target.value})}
                placeholder="Enter any notes for today (optional)"
              />
            </Form.Group>
            
            <div className="d-flex justify-content-end">
              <Button variant="secondary" className="me-2" onClick={() => setShowOpenDayModal(false)}>
                Cancel
              </Button>
              <Button 
                type="submit" 
                variant="success"
                disabled={loading}
              >
                {loading ? (
                  <>
                    <span className="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                    Opening...
                  </>
                ) : (
                  <>
                    <i className="fas fa-play me-1"></i>
                    Open Day
                  </>
                )}
              </Button>
            </div>
          </Form>
        </Modal.Body>
      </Modal>

      {/* Close Day Modal */}
      <Modal show={showCloseDayModal} onHide={() => setShowCloseDayModal(false)} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-stop me-2" style={{ color: operatorColor }}></i>
            Close Day - {operatorName}
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form onSubmit={handleCloseDay}>
            {dailySummary && (
              <div className="mb-3 p-3 bg-light rounded">
                <h6>Today's Summary</h6>
                <Row className="g-2">
                  <Col md={6}>
                    <small className="text-muted">Opening Balance:</small>
                    <div><strong>{formatCurrency(dailySummary.openingBalance)}</strong></div>
                  </Col>
                  <Col md={6}>
                    <small className="text-muted">Total Transactions:</small>
                    <div><strong>{dailySummary.transactionCount}</strong></div>
                  </Col>
                  <Col md={6}>
                    <small className="text-muted">Total Recharge:</small>
                    <div><strong>{formatCurrency(dailySummary.totalRecharge)}</strong></div>
                  </Col>
                  <Col md={6}>
                    <small className="text-muted">Total Commission:</small>
                    <div><strong>{formatCurrency(dailySummary.totalCommission)}</strong></div>
                  </Col>
                </Row>
              </div>
            )}
            
            <Form.Group className="mb-3">
              <Form.Label>Closing Balance <span className="text-danger">*</span></Form.Label>
              <Form.Control
                type="number"
                step="0.01"
                name="closingBalance"
                value={closeDayForm.closingBalance}
                onChange={(e) => setCloseDayForm({...closeDayForm, closingBalance: e.target.value})}
                placeholder="Enter closing balance"
                required
              />
              <Form.Text className="text-muted">
                Enter the final balance after today's operations
              </Form.Text>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Notes</Form.Label>
              <Form.Control
                as="textarea"
                rows={3}
                name="notes"
                value={closeDayForm.notes}
                onChange={(e) => setCloseDayForm({...closeDayForm, notes: e.target.value})}
                placeholder="Enter any closing notes (optional)"
              />
            </Form.Group>
            
            <div className="d-flex justify-content-end">
              <Button variant="secondary" className="me-2" onClick={() => setShowCloseDayModal(false)}>
                Cancel
              </Button>
              <Button 
                type="submit" 
                variant="danger"
                disabled={loading}
              >
                {loading ? (
                  <>
                    <span className="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                    Closing...
                  </>
                ) : (
                  <>
                    <i className="fas fa-stop me-1"></i>
                    Close Day
                  </>
                )}
              </Button>
            </div>
          </Form>
        </Modal.Body>
      </Modal>
    </div>
  );
};

export default MobileRechargeOperator;