import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Table, Modal, Form, Alert, Badge, InputGroup } from 'react-bootstrap';
import { FaSearch, FaMoneyBillWave, FaPrint, FaFileInvoice, FaCalendarAlt, FaFilter } from 'react-icons/fa';
import './BillCollection.css';

const BillCollection = () => {
  const [bills, setBills] = useState([]);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [selectedBill, setSelectedBill] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedArea, setSelectedArea] = useState('all');
  const [selectedStatus, setSelectedStatus] = useState('all');
  const [showAlert, setShowAlert] = useState({ show: false, message: '', type: '' });
  const [paymentData, setPaymentData] = useState({
    paymentMethod: 'cash',
    amountPaid: '',
    discount: 0,
    notes: ''
  });

  // Mock data for bills
  useEffect(() => {
    const mockBills = [
      {
        id: 1,
        customerName: 'আব্দুল করিম',
        meterNumber: 'PB001234',
        phoneNumber: '01712345678',
        area: 'রামপুর',
        billMonth: '2024-01',
        unitsConsumed: 150,
        billAmount: 825,
        dueDate: '2024-02-15',
        status: 'pending',
        connectionType: 'residential',
        previousReading: 1200,
        currentReading: 1350,
        tariffRate: 5.5
      },
      {
        id: 2,
        customerName: 'ফাতেমা খাতুন',
        meterNumber: 'PB001235',
        phoneNumber: '01812345679',
        area: 'রামপুর',
        billMonth: '2024-01',
        unitsConsumed: 120,
        billAmount: 660,
        dueDate: '2024-02-15',
        status: 'pending',
        connectionType: 'residential',
        previousReading: 800,
        currentReading: 920,
        tariffRate: 5.5
      },
      {
        id: 3,
        customerName: 'মোহাম্মদ আলী',
        meterNumber: 'PB001236',
        phoneNumber: '01912345680',
        area: 'কালিগঞ্জ',
        billMonth: '2024-01',
        unitsConsumed: 300,
        billAmount: 2550,
        dueDate: '2024-02-15',
        status: 'overdue',
        connectionType: 'commercial',
        previousReading: 2500,
        currentReading: 2800,
        tariffRate: 8.5
      },
      {
        id: 4,
        customerName: 'রহিমা বেগম',
        meterNumber: 'PB001237',
        phoneNumber: '01612345681',
        area: 'রামপুর',
        billMonth: '2024-01',
        unitsConsumed: 80,
        billAmount: 440,
        dueDate: '2024-02-15',
        status: 'paid',
        connectionType: 'residential',
        previousReading: 600,
        currentReading: 680,
        tariffRate: 5.5,
        paidDate: '2024-02-10',
        paidAmount: 440
      }
    ];
    setBills(mockBills);
  }, []);

  // Filter bills
  const filteredBills = bills.filter(bill => {
    const matchesSearch = bill.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         bill.meterNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         bill.phoneNumber.includes(searchTerm);
    const matchesArea = selectedArea === 'all' || bill.area === selectedArea;
    const matchesStatus = selectedStatus === 'all' || bill.status === selectedStatus;
    return matchesSearch && matchesArea && matchesStatus;
  });

  // Calculate statistics
  const totalBills = filteredBills.length;
  const pendingBills = filteredBills.filter(bill => bill.status === 'pending').length;
  const overdueBills = filteredBills.filter(bill => bill.status === 'overdue').length;
  const paidBills = filteredBills.filter(bill => bill.status === 'paid').length;
  const totalAmount = filteredBills.reduce((sum, bill) => sum + bill.billAmount, 0);
  const collectedAmount = filteredBills.filter(bill => bill.status === 'paid').reduce((sum, bill) => sum + bill.billAmount, 0);

  // Handle payment
  const handlePayment = (bill) => {
    setSelectedBill(bill);
    setPaymentData({
      paymentMethod: 'cash',
      amountPaid: bill.billAmount.toString(),
      discount: 0,
      notes: ''
    });
    setShowPaymentModal(true);
  };

  // Process payment
  const processPayment = (e) => {
    e.preventDefault();
    
    const updatedBills = bills.map(bill => 
      bill.id === selectedBill.id 
        ? {
            ...bill,
            status: 'paid',
            paidDate: new Date().toISOString().split('T')[0],
            paidAmount: parseFloat(paymentData.amountPaid),
            paymentMethod: paymentData.paymentMethod,
            discount: parseFloat(paymentData.discount),
            notes: paymentData.notes
          }
        : bill
    );
    
    setBills(updatedBills);
    setShowAlert({ 
      show: true, 
      message: `${selectedBill.customerName} এর বিল সফলভাবে পরিশোধ করা হয়েছে।`, 
      type: 'success' 
    });
    
    setShowPaymentModal(false);
    setSelectedBill(null);
    setTimeout(() => setShowAlert({ show: false, message: '', type: '' }), 3000);
  };

  // Get status badge variant
  const getStatusBadge = (status) => {
    switch (status) {
      case 'paid': return { bg: 'success', text: 'পরিশোধিত' };
      case 'pending': return { bg: 'warning', text: 'অপেক্ষমান' };
      case 'overdue': return { bg: 'danger', text: 'বকেয়া' };
      default: return { bg: 'secondary', text: 'অজানা' };
    }
  };

  // Check if bill is overdue
  const isOverdue = (dueDate, status) => {
    if (status === 'paid') return false;
    return new Date(dueDate) < new Date();
  };

  // Print bill
  const printBill = (bill) => {
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
      <html>
        <head>
          <title>বিদ্যুৎ বিল - ${bill.meterNumber}</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            .header { text-align: center; margin-bottom: 30px; }
            .bill-details { margin-bottom: 20px; }
            .table { width: 100%; border-collapse: collapse; }
            .table th, .table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            .table th { background-color: #f2f2f2; }
            .total { font-weight: bold; font-size: 18px; }
          </style>
        </head>
        <body>
          <div class="header">
            <h2>পল্লী বিদ্যুৎ সমিতি</h2>
            <h3>বিদ্যুৎ বিল</h3>
          </div>
          <div class="bill-details">
            <p><strong>গ্রাহকের নাম:</strong> ${bill.customerName}</p>
            <p><strong>মিটার নম্বর:</strong> ${bill.meterNumber}</p>
            <p><strong>বিলের মাস:</strong> ${new Date(bill.billMonth).toLocaleDateString('bn-BD', { year: 'numeric', month: 'long' })}</p>
            <p><strong>এলাকা:</strong> ${bill.area}</p>
          </div>
          <table class="table">
            <tr><th>বিবরণ</th><th>পরিমাণ</th></tr>
            <tr><td>পূর্ববর্তী রিডিং</td><td>${bill.previousReading} ইউনিট</td></tr>
            <tr><td>বর্তমান রিডিং</td><td>${bill.currentReading} ইউনিট</td></tr>
            <tr><td>ব্যবহৃত ইউনিট</td><td>${bill.unitsConsumed} ইউনিট</td></tr>
            <tr><td>ট্যারিফ রেট</td><td>৳${bill.tariffRate}/ইউনিট</td></tr>
            <tr class="total"><td>মোট বিল</td><td>৳${bill.billAmount}</td></tr>
          </table>
          <p><strong>পরিশোধের শেষ তারিখ:</strong> ${new Date(bill.dueDate).toLocaleDateString('bn-BD')}</p>
        </body>
      </html>
    `);
    printWindow.document.close();
    printWindow.print();
  };

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h2>পল্লী বিদ্যুৎ - বিল সংগ্রহ</h2>
        </Col>
      </Row>

      {showAlert.show && (
        <Alert variant={showAlert.type} className="mb-3">
          {showAlert.message}
        </Alert>
      )}

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card total">
            <Card.Body className="text-center">
              <h3>{totalBills}</h3>
              <p>মোট বিল</p>
              <small>৳{totalAmount.toLocaleString()}</small>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card pending">
            <Card.Body className="text-center">
              <h3>{pendingBills}</h3>
              <p>অপেক্ষমান</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card overdue">
            <Card.Body className="text-center">
              <h3>{overdueBills}</h3>
              <p>বকেয়া</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card collected">
            <Card.Body className="text-center">
              <h3>{paidBills}</h3>
              <p>সংগৃহীত</p>
              <small>৳{collectedAmount.toLocaleString()}</small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Filters */}
      <Row className="mb-3">
        <Col md={3}>
          <InputGroup>
            <Form.Control
              type="text"
              placeholder="গ্রাহক খুঁজুন..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <Button variant="outline-secondary">
              <FaSearch />
            </Button>
          </InputGroup>
        </Col>
        <Col md={2}>
          <Form.Select value={selectedArea} onChange={(e) => setSelectedArea(e.target.value)}>
            <option value="all">সব এলাকা</option>
            <option value="রামপুর">রামপুর</option>
            <option value="কালিগঞ্জ">কালিগঞ্জ</option>
            <option value="বাজারপাড়া">বাজারপাড়া</option>
          </Form.Select>
        </Col>
        <Col md={2}>
          <Form.Select value={selectedStatus} onChange={(e) => setSelectedStatus(e.target.value)}>
            <option value="all">সব অবস্থা</option>
            <option value="pending">অপেক্ষমান</option>
            <option value="overdue">বকেয়া</option>
            <option value="paid">পরিশোধিত</option>
          </Form.Select>
        </Col>
        <Col md={5} className="text-end">
          <Button variant="info" className="me-2">
            <FaCalendarAlt className="me-1" /> আজকের সংগ্রহ
          </Button>
          <Button variant="success">
            <FaFileInvoice className="me-1" /> রিপোর্ট
          </Button>
        </Col>
      </Row>

      {/* Bills Table */}
      <Card>
        <Card.Body>
          <Table responsive striped hover>
            <thead>
              <tr>
                <th>গ্রাহকের তথ্য</th>
                <th>বিলের মাস</th>
                <th>ইউনিট</th>
                <th>বিলের পরিমাণ</th>
                <th>শেষ তারিখ</th>
                <th>অবস্থা</th>
                <th>কার্যক্রম</th>
              </tr>
            </thead>
            <tbody>
              {filteredBills.map(bill => {
                const statusBadge = getStatusBadge(bill.status);
                const overdue = isOverdue(bill.dueDate, bill.status);
                
                return (
                  <tr key={bill.id} className={overdue ? 'table-danger' : ''}>
                    <td>
                      <strong>{bill.customerName}</strong>
                      <br />
                      <small className="text-muted">{bill.meterNumber}</small>
                      <br />
                      <small className="text-muted">{bill.phoneNumber}</small>
                      <br />
                      <small className="text-muted">{bill.area}</small>
                    </td>
                    <td>
                      {new Date(bill.billMonth).toLocaleDateString('bn-BD', { 
                        year: 'numeric', 
                        month: 'long' 
                      })}
                    </td>
                    <td>
                      <div>
                        <small>পূর্ববর্তী: {bill.previousReading}</small><br />
                        <small>বর্তমান: {bill.currentReading}</small><br />
                        <strong>ব্যবহৃত: {bill.unitsConsumed}</strong>
                      </div>
                    </td>
                    <td>
                      <strong className="text-primary">৳{bill.billAmount}</strong>
                      <br />
                      <small className="text-muted">
                        {bill.connectionType === 'residential' ? 'আবাসিক' : 'বাণিজ্যিক'}
                      </small>
                    </td>
                    <td>
                      <div className={overdue ? 'text-danger' : ''}>
                        {new Date(bill.dueDate).toLocaleDateString('bn-BD')}
                        {overdue && (
                          <>
                            <br />
                            <small><strong>মেয়াদোত্তীর্ণ</strong></small>
                          </>
                        )}
                      </div>
                    </td>
                    <td>
                      <Badge bg={statusBadge.bg}>
                        {statusBadge.text}
                      </Badge>
                      {bill.status === 'paid' && (
                        <div className="mt-1">
                          <small className="text-success">
                            পরিশোধ: {new Date(bill.paidDate).toLocaleDateString('bn-BD')}
                          </small>
                        </div>
                      )}
                    </td>
                    <td>
                      <div className="d-flex flex-column gap-1">
                        {bill.status !== 'paid' && (
                          <Button 
                            variant="success" 
                            size="sm" 
                            onClick={() => handlePayment(bill)}
                          >
                            <FaMoneyBillWave className="me-1" /> পরিশোধ
                          </Button>
                        )}
                        <Button 
                          variant="outline-info" 
                          size="sm" 
                          onClick={() => printBill(bill)}
                        >
                          <FaPrint className="me-1" /> প্রিন্ট
                        </Button>
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </Table>

          {filteredBills.length === 0 && (
            <div className="text-center py-4">
              <p className="text-muted">কোন বিল পাওয়া যায়নি।</p>
            </div>
          )}
        </Card.Body>
      </Card>

      {/* Payment Modal */}
      <Modal show={showPaymentModal} onHide={() => setShowPaymentModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>বিল পরিশোধ</Modal.Title>
        </Modal.Header>
        {selectedBill && (
          <Form onSubmit={processPayment}>
            <Modal.Body>
              <Row className="mb-3">
                <Col md={6}>
                  <h5>গ্রাহকের তথ্য</h5>
                  <p><strong>নাম:</strong> {selectedBill.customerName}</p>
                  <p><strong>মিটার:</strong> {selectedBill.meterNumber}</p>
                  <p><strong>এলাকা:</strong> {selectedBill.area}</p>
                </Col>
                <Col md={6}>
                  <h5>বিলের তথ্য</h5>
                  <p><strong>মাস:</strong> {new Date(selectedBill.billMonth).toLocaleDateString('bn-BD', { year: 'numeric', month: 'long' })}</p>
                  <p><strong>ইউনিট:</strong> {selectedBill.unitsConsumed}</p>
                  <p><strong>মোট বিল:</strong> ৳{selectedBill.billAmount}</p>
                </Col>
              </Row>
              
              <hr />
              
              <Row>
                <Col md={6}>
                  <Form.Group className="mb-3">
                    <Form.Label>পরিশোধের পদ্ধতি</Form.Label>
                    <Form.Select
                      value={paymentData.paymentMethod}
                      onChange={(e) => setPaymentData({...paymentData, paymentMethod: e.target.value})}
                    >
                      <option value="cash">নগদ</option>
                      <option value="bkash">বিকাশ</option>
                      <option value="nagad">নগদ (মোবাইল)</option>
                      <option value="bank">ব্যাংক</option>
                    </Form.Select>
                  </Form.Group>
                </Col>
                <Col md={6}>
                  <Form.Group className="mb-3">
                    <Form.Label>পরিশোধিত পরিমাণ</Form.Label>
                    <Form.Control
                      type="number"
                      value={paymentData.amountPaid}
                      onChange={(e) => setPaymentData({...paymentData, amountPaid: e.target.value})}
                      required
                    />
                  </Form.Group>
                </Col>
              </Row>
              
              <Row>
                <Col md={6}>
                  <Form.Group className="mb-3">
                    <Form.Label>ছাড় (যদি থাকে)</Form.Label>
                    <Form.Control
                      type="number"
                      value={paymentData.discount}
                      onChange={(e) => setPaymentData({...paymentData, discount: e.target.value})}
                    />
                  </Form.Group>
                </Col>
                <Col md={6}>
                  <Form.Group className="mb-3">
                    <Form.Label>মন্তব্য</Form.Label>
                    <Form.Control
                      as="textarea"
                      rows={2}
                      value={paymentData.notes}
                      onChange={(e) => setPaymentData({...paymentData, notes: e.target.value})}
                      placeholder="অতিরিক্ত মন্তব্য..."
                    />
                  </Form.Group>
                </Col>
              </Row>
              
              <div className="payment-summary p-3 bg-light rounded">
                <h6>পরিশোধের সারসংক্ষেপ</h6>
                <div className="d-flex justify-content-between">
                  <span>মূল বিল:</span>
                  <span>৳{selectedBill.billAmount}</span>
                </div>
                <div className="d-flex justify-content-between">
                  <span>ছাড়:</span>
                  <span>৳{paymentData.discount || 0}</span>
                </div>
                <hr />
                <div className="d-flex justify-content-between fw-bold">
                  <span>মোট পরিশোধ:</span>
                  <span>৳{(parseFloat(paymentData.amountPaid) || 0) - (parseFloat(paymentData.discount) || 0)}</span>
                </div>
              </div>
            </Modal.Body>
            <Modal.Footer>
              <Button variant="secondary" onClick={() => setShowPaymentModal(false)}>
                বাতিল
              </Button>
              <Button variant="success" type="submit">
                <FaMoneyBillWave className="me-1" /> পরিশোধ নিশ্চিত করুন
              </Button>
            </Modal.Footer>
          </Form>
        )}
      </Modal>
    </Container>
  );
};

export default BillCollection;