import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Table, Button, Badge, Form, Modal, Alert, ProgressBar } from 'react-bootstrap';
import { FaBell, FaCalendarAlt, FaExclamationTriangle, FaPhone, FaSms, FaEnvelope, FaPrint, FaDownload, FaSearch, FaFilter } from 'react-icons/fa';
import './DueDateTracking.css';

const DueDateTracking = () => {
  const [bills, setBills] = useState([]);
  const [filteredBills, setFilteredBills] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedArea, setSelectedArea] = useState('');
  const [selectedPriority, setSelectedPriority] = useState('');
  const [showReminderModal, setShowReminderModal] = useState(false);
  const [selectedBill, setSelectedBill] = useState(null);
  const [reminderType, setReminderType] = useState('');
  const [showAlert, setShowAlert] = useState(false);
  const [alertMessage, setAlertMessage] = useState('');

  // Mock data for bills with due dates
  useEffect(() => {
    const mockBills = [
      {
        id: 1,
        customerName: 'আহমেদ হাসান',
        customerId: 'PB001',
        area: 'ঢাকা',
        billAmount: 1250,
        dueDate: '2024-01-25',
        daysOverdue: 5,
        priority: 'high',
        phone: '01712345678',
        email: 'ahmed@email.com',
        address: 'ধানমন্ডি, ঢাকা',
        lastPayment: '2023-12-15'
      },
      {
        id: 2,
        customerName: 'ফাতেমা খাতুন',
        customerId: 'PB002',
        area: 'চট্টগ্রাম',
        billAmount: 980,
        dueDate: '2024-01-28',
        daysOverdue: 2,
        priority: 'medium',
        phone: '01823456789',
        email: 'fatema@email.com',
        address: 'আগ্রাবাদ, চট্টগ্রাম',
        lastPayment: '2023-12-20'
      },
      {
        id: 3,
        customerName: 'মোহাম্মদ রহিম',
        customerId: 'PB003',
        area: 'সিলেট',
        billAmount: 1450,
        dueDate: '2024-01-30',
        daysOverdue: 0,
        priority: 'low',
        phone: '01934567890',
        email: 'rahim@email.com',
        address: 'জিন্দাবাজার, সিলেট',
        lastPayment: '2023-12-25'
      },
      {
        id: 4,
        customerName: 'সালমা বেগম',
        customerId: 'PB004',
        area: 'ঢাকা',
        billAmount: 2100,
        dueDate: '2024-01-22',
        daysOverdue: 8,
        priority: 'high',
        phone: '01645678901',
        email: 'salma@email.com',
        address: 'গুলশান, ঢাকা',
        lastPayment: '2023-12-10'
      },
      {
        id: 5,
        customerName: 'করিম উদ্দিন',
        customerId: 'PB005',
        area: 'রাজশাহী',
        billAmount: 875,
        dueDate: '2024-02-01',
        daysOverdue: -2,
        priority: 'low',
        phone: '01756789012',
        email: 'karim@email.com',
        address: 'বোয়ালিয়া, রাজশাহী',
        lastPayment: '2023-12-28'
      }
    ];
    setBills(mockBills);
    setFilteredBills(mockBills);
  }, []);

  // Filter bills based on search and filters
  useEffect(() => {
    let filtered = bills.filter(bill => {
      const matchesSearch = bill.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           bill.customerId.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesArea = selectedArea === '' || bill.area === selectedArea;
      const matchesPriority = selectedPriority === '' || bill.priority === selectedPriority;
      
      return matchesSearch && matchesArea && matchesPriority;
    });
    setFilteredBills(filtered);
  }, [searchTerm, selectedArea, selectedPriority, bills]);

  const getPriorityBadge = (priority) => {
    const variants = {
      high: 'danger',
      medium: 'warning',
      low: 'success'
    };
    const labels = {
      high: 'উচ্চ',
      medium: 'মধ্যম',
      low: 'নিম্ন'
    };
    return <Badge bg={variants[priority]}>{labels[priority]}</Badge>;
  };

  const getDaysOverdueBadge = (days) => {
    if (days > 7) return <Badge bg="danger">অতিরিক্ত বিলম্ব ({days} দিন)</Badge>;
    if (days > 0) return <Badge bg="warning">বিলম্বিত ({days} দিন)</Badge>;
    if (days === 0) return <Badge bg="info">আজ শেষ দিন</Badge>;
    return <Badge bg="success">আগামী {Math.abs(days)} দিন</Badge>;
  };

  const handleSendReminder = (bill, type) => {
    setSelectedBill(bill);
    setReminderType(type);
    setShowReminderModal(true);
  };

  const confirmSendReminder = () => {
    const typeLabels = {
      sms: 'SMS',
      call: 'ফোন কল',
      email: 'ইমেইল'
    };
    
    setAlertMessage(`${selectedBill.customerName} এর কাছে ${typeLabels[reminderType]} রিমাইন্ডার পাঠানো হয়েছে।`);
    setShowAlert(true);
    setShowReminderModal(false);
    
    setTimeout(() => setShowAlert(false), 3000);
  };

  const exportToCSV = () => {
    const csvContent = [
      ['গ্রাহক নাম', 'গ্রাহক আইডি', 'এলাকা', 'বিল পরিমাণ', 'শেষ তারিখ', 'বিলম্ব', 'অগ্রাধিকার'],
      ...filteredBills.map(bill => [
        bill.customerName,
        bill.customerId,
        bill.area,
        bill.billAmount,
        bill.dueDate,
        bill.daysOverdue > 0 ? `${bill.daysOverdue} দিন` : 'না',
        bill.priority === 'high' ? 'উচ্চ' : bill.priority === 'medium' ? 'মধ্যম' : 'নিম্ন'
      ])
    ].map(row => row.join(',')).join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'due-date-tracking.csv';
    a.click();
  };

  const printReport = () => {
    window.print();
  };

  const areas = [...new Set(bills.map(bill => bill.area))];
  const overdueBills = filteredBills.filter(bill => bill.daysOverdue > 0);
  const todayDueBills = filteredBills.filter(bill => bill.daysOverdue === 0);
  const upcomingBills = filteredBills.filter(bill => bill.daysOverdue < 0);

  return (
    <Container fluid className="due-date-tracking-container">
      {showAlert && (
        <Alert variant="success" className="alert-notification">
          {alertMessage}
        </Alert>
      )}

      <Row className="mb-4">
        <Col>
          <h2 className="page-title">
            <FaCalendarAlt className="me-2" />
            শেষ তারিখ ট্র্যাকিং
          </h2>
        </Col>
      </Row>

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card overdue-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">বিলম্বিত বিল</h5>
                  <h3 className="card-value">{overdueBills.length}</h3>
                </div>
                <FaExclamationTriangle className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card today-due-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">আজ শেষ দিন</h5>
                  <h3 className="card-value">{todayDueBills.length}</h3>
                </div>
                <FaBell className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card upcoming-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">আগামী বিল</h5>
                  <h3 className="card-value">{upcomingBills.length}</h3>
                </div>
                <FaCalendarAlt className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card total-amount-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">মোট বিলম্বিত টাকা</h5>
                  <h3 className="card-value">৳{overdueBills.reduce((sum, bill) => sum + bill.billAmount, 0)}</h3>
                </div>
                <FaExclamationTriangle className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Filters */}
      <Card className="mb-4">
        <Card.Body>
          <Row>
            <Col md={4}>
              <Form.Group>
                <Form.Label>অনুসন্ধান</Form.Label>
                <div className="search-input-wrapper">
                  <FaSearch className="search-icon" />
                  <Form.Control
                    type="text"
                    placeholder="গ্রাহক নাম বা আইডি দিয়ে অনুসন্ধান করুন..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                  />
                </div>
              </Form.Group>
            </Col>
            <Col md={3}>
              <Form.Group>
                <Form.Label>এলাকা</Form.Label>
                <Form.Select value={selectedArea} onChange={(e) => setSelectedArea(e.target.value)}>
                  <option value="">সব এলাকা</option>
                  {areas.map(area => (
                    <option key={area} value={area}>{area}</option>
                  ))}
                </Form.Select>
              </Form.Group>
            </Col>
            <Col md={3}>
              <Form.Group>
                <Form.Label>অগ্রাধিকার</Form.Label>
                <Form.Select value={selectedPriority} onChange={(e) => setSelectedPriority(e.target.value)}>
                  <option value="">সব অগ্রাধিকার</option>
                  <option value="high">উচ্চ</option>
                  <option value="medium">মধ্যম</option>
                  <option value="low">নিম্ন</option>
                </Form.Select>
              </Form.Group>
            </Col>
            <Col md={2} className="d-flex align-items-end">
              <div className="action-buttons">
                <Button variant="outline-primary" onClick={exportToCSV} className="me-2">
                  <FaDownload /> CSV
                </Button>
                <Button variant="outline-secondary" onClick={printReport}>
                  <FaPrint /> প্রিন্ট
                </Button>
              </div>
            </Col>
          </Row>
        </Card.Body>
      </Card>

      {/* Bills Table */}
      <Card>
        <Card.Header>
          <h5 className="mb-0">বিলের তালিকা ({filteredBills.length}টি)</h5>
        </Card.Header>
        <Card.Body className="p-0">
          <div className="table-responsive">
            <Table striped hover className="mb-0">
              <thead className="table-dark">
                <tr>
                  <th>গ্রাহক তথ্য</th>
                  <th>এলাকা</th>
                  <th>বিল পরিমাণ</th>
                  <th>শেষ তারিখ</th>
                  <th>অবস্থা</th>
                  <th>অগ্রাধিকার</th>
                  <th>রিমাইন্ডার</th>
                </tr>
              </thead>
              <tbody>
                {filteredBills.map(bill => (
                  <tr key={bill.id}>
                    <td>
                      <div>
                        <strong>{bill.customerName}</strong>
                        <br />
                        <small className="text-muted">{bill.customerId}</small>
                        <br />
                        <small className="text-muted">{bill.phone}</small>
                      </div>
                    </td>
                    <td>{bill.area}</td>
                    <td>
                      <strong>৳{bill.billAmount}</strong>
                    </td>
                    <td>{bill.dueDate}</td>
                    <td>{getDaysOverdueBadge(bill.daysOverdue)}</td>
                    <td>{getPriorityBadge(bill.priority)}</td>
                    <td>
                      <div className="reminder-buttons">
                        <Button
                          size="sm"
                          variant="outline-primary"
                          onClick={() => handleSendReminder(bill, 'sms')}
                          className="me-1 mb-1"
                        >
                          <FaSms />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline-success"
                          onClick={() => handleSendReminder(bill, 'call')}
                          className="me-1 mb-1"
                        >
                          <FaPhone />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline-info"
                          onClick={() => handleSendReminder(bill, 'email')}
                          className="mb-1"
                        >
                          <FaEnvelope />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </Table>
          </div>
        </Card.Body>
      </Card>

      {/* Reminder Modal */}
      <Modal show={showReminderModal} onHide={() => setShowReminderModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>রিমাইন্ডার পাঠান</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          {selectedBill && (
            <div>
              <p><strong>গ্রাহক:</strong> {selectedBill.customerName}</p>
              <p><strong>বিল পরিমাণ:</strong> ৳{selectedBill.billAmount}</p>
              <p><strong>শেষ তারিখ:</strong> {selectedBill.dueDate}</p>
              <p><strong>রিমাইন্ডার টাইপ:</strong> {
                reminderType === 'sms' ? 'SMS' :
                reminderType === 'call' ? 'ফোন কল' : 'ইমেইল'
              }</p>
              <hr />
              <p>আপনি কি নিশ্চিত যে এই গ্রাহকের কাছে রিমাইন্ডার পাঠাতে চান?</p>
            </div>
          )}
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowReminderModal(false)}>
            বাতিল
          </Button>
          <Button variant="primary" onClick={confirmSendReminder}>
            রিমাইন্ডার পাঠান
          </Button>
        </Modal.Footer>
      </Modal>
    </Container>
  );
};

export default DueDateTracking;