import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Form, Button, Table, Badge } from 'react-bootstrap';
import { FaFileAlt, FaDownload, FaPrint, FaChartBar, FaCalendarAlt, FaUsers, FaMoneyBillWave } from 'react-icons/fa';
import { Chart as ChartJS, CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend, ArcElement, LineElement, PointElement } from 'chart.js';
import { Bar, Pie, Line } from 'react-chartjs-2';
import './PalliBidyutReports.css';

ChartJS.register(
  CategoryScale,
  LinearScale,
  BarElement,
  Title,
  Tooltip,
  Legend,
  ArcElement,
  LineElement,
  PointElement
);

const PalliBidyutReports = () => {
  const [reportType, setReportType] = useState('monthly');
  const [selectedMonth, setSelectedMonth] = useState(new Date().getMonth() + 1);
  const [selectedYear, setSelectedYear] = useState(new Date().getFullYear());
  const [reportData, setReportData] = useState({});

  // Mock data for reports
  useEffect(() => {
    const mockData = {
      monthly: {
        totalCustomers: 1250,
        totalBills: 1180,
        totalCollection: 2450000,
        pendingBills: 70,
        pendingAmount: 125000,
        collectionRate: 94.4,
        areaWiseData: [
          { area: 'ঢাকা', customers: 450, collection: 980000, pending: 25 },
          { area: 'চট্টগ্রাম', customers: 320, collection: 720000, pending: 18 },
          { area: 'সিলেট', customers: 280, collection: 450000, pending: 15 },
          { area: 'রাজশাহী', customers: 200, collection: 300000, pending: 12 }
        ],
        dailyCollection: [
          { date: '01', amount: 85000 },
          { date: '02', amount: 92000 },
          { date: '03', amount: 78000 },
          { date: '04', amount: 105000 },
          { date: '05', amount: 88000 },
          { date: '06', amount: 95000 },
          { date: '07', amount: 110000 }
        ],
        billStatus: {
          paid: 1180,
          pending: 45,
          overdue: 25
        }
      }
    };
    setReportData(mockData);
  }, []);

  const generateReport = () => {
    // Mock report generation
    console.log(`Generating ${reportType} report for ${selectedMonth}/${selectedYear}`);
  };

  const exportToPDF = () => {
    window.print();
  };

  const exportToExcel = () => {
    // Mock Excel export
    const csvContent = [
      ['এলাকা', 'গ্রাহক সংখ্যা', 'আদায়', 'বকেয়া'],
      ...reportData.monthly?.areaWiseData?.map(area => [
        area.area,
        area.customers,
        area.collection,
        area.pending
      ]) || []
    ].map(row => row.join(',')).join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'palli-bidyut-report.csv';
    a.click();
  };

  // Chart configurations
  const barChartData = {
    labels: reportData.monthly?.areaWiseData?.map(area => area.area) || [],
    datasets: [
      {
        label: 'আদায় (টাকা)',
        data: reportData.monthly?.areaWiseData?.map(area => area.collection) || [],
        backgroundColor: 'rgba(54, 162, 235, 0.6)',
        borderColor: 'rgba(54, 162, 235, 1)',
        borderWidth: 1
      }
    ]
  };

  const pieChartData = {
    labels: ['পরিশোধিত', 'বকেয়া', 'অতিরিক্ত বকেয়া'],
    datasets: [
      {
        data: [
          reportData.monthly?.billStatus?.paid || 0,
          reportData.monthly?.billStatus?.pending || 0,
          reportData.monthly?.billStatus?.overdue || 0
        ],
        backgroundColor: [
          '#28a745',
          '#ffc107',
          '#dc3545'
        ],
        borderWidth: 2
      }
    ]
  };

  const lineChartData = {
    labels: reportData.monthly?.dailyCollection?.map(day => day.date) || [],
    datasets: [
      {
        label: 'দৈনিক আদায়',
        data: reportData.monthly?.dailyCollection?.map(day => day.amount) || [],
        borderColor: 'rgb(75, 192, 192)',
        backgroundColor: 'rgba(75, 192, 192, 0.2)',
        tension: 0.1
      }
    ]
  };

  const chartOptions = {
    responsive: true,
    plugins: {
      legend: {
        position: 'top',
      },
      title: {
        display: true,
        text: 'পল্লী বিদ্যুৎ রিপোর্ট'
      }
    }
  };

  return (
    <Container fluid className="palli-bidyut-reports-container">
      <Row className="mb-4">
        <Col>
          <h2 className="page-title">
            <FaFileAlt className="me-2" />
            পল্লী বিদ্যুৎ রিপোর্ট
          </h2>
        </Col>
      </Row>

      {/* Report Filters */}
      <Card className="mb-4">
        <Card.Header>
          <h5 className="mb-0">রিপোর্ট ফিল্টার</h5>
        </Card.Header>
        <Card.Body>
          <Row>
            <Col md={3}>
              <Form.Group>
                <Form.Label>রিপোর্ট টাইপ</Form.Label>
                <Form.Select value={reportType} onChange={(e) => setReportType(e.target.value)}>
                  <option value="daily">দৈনিক</option>
                  <option value="weekly">সাপ্তাহিক</option>
                  <option value="monthly">মাসিক</option>
                  <option value="yearly">বার্ষিক</option>
                </Form.Select>
              </Form.Group>
            </Col>
            <Col md={3}>
              <Form.Group>
                <Form.Label>মাস</Form.Label>
                <Form.Select value={selectedMonth} onChange={(e) => setSelectedMonth(e.target.value)}>
                  <option value="1">জানুয়ারি</option>
                  <option value="2">ফেব্রুয়ারি</option>
                  <option value="3">মার্চ</option>
                  <option value="4">এপ্রিল</option>
                  <option value="5">মে</option>
                  <option value="6">জুন</option>
                  <option value="7">জুলাই</option>
                  <option value="8">আগস্ট</option>
                  <option value="9">সেপ্টেম্বর</option>
                  <option value="10">অক্টোবর</option>
                  <option value="11">নভেম্বর</option>
                  <option value="12">ডিসেম্বর</option>
                </Form.Select>
              </Form.Group>
            </Col>
            <Col md={3}>
              <Form.Group>
                <Form.Label>বছর</Form.Label>
                <Form.Select value={selectedYear} onChange={(e) => setSelectedYear(e.target.value)}>
                  <option value="2024">২০২৪</option>
                  <option value="2023">২০২৩</option>
                  <option value="2022">২০২২</option>
                </Form.Select>
              </Form.Group>
            </Col>
            <Col md={3} className="d-flex align-items-end">
              <div className="action-buttons w-100">
                <Button variant="primary" onClick={generateReport} className="me-2">
                  <FaChartBar /> রিপোর্ট তৈরি করুন
                </Button>
              </div>
            </Col>
          </Row>
        </Card.Body>
      </Card>

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card customers-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">মোট গ্রাহক</h5>
                  <h3 className="card-value">{reportData.monthly?.totalCustomers || 0}</h3>
                </div>
                <FaUsers className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card collection-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">মোট আদায়</h5>
                  <h3 className="card-value">৳{reportData.monthly?.totalCollection?.toLocaleString() || 0}</h3>
                </div>
                <FaMoneyBillWave className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card pending-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">বকেয়া বিল</h5>
                  <h3 className="card-value">{reportData.monthly?.pendingBills || 0}</h3>
                </div>
                <FaCalendarAlt className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card rate-card">
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <div>
                  <h5 className="card-title">আদায়ের হার</h5>
                  <h3 className="card-value">{reportData.monthly?.collectionRate || 0}%</h3>
                </div>
                <FaChartBar className="card-icon" />
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Charts */}
      <Row className="mb-4">
        <Col md={8}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">এলাকাভিত্তিক আদায়</h5>
            </Card.Header>
            <Card.Body>
              <Bar data={barChartData} options={chartOptions} />
            </Card.Body>
          </Card>
        </Col>
        <Col md={4}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">বিলের অবস্থা</h5>
            </Card.Header>
            <Card.Body>
              <Pie data={pieChartData} options={chartOptions} />
            </Card.Body>
          </Card>
        </Col>
      </Row>

      <Row className="mb-4">
        <Col>
          <Card>
            <Card.Header>
              <h5 className="mb-0">দৈনিক আদায়ের ট্রেন্ড</h5>
            </Card.Header>
            <Card.Body>
              <Line data={lineChartData} options={chartOptions} />
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Area-wise Report Table */}
      <Card className="mb-4">
        <Card.Header className="d-flex justify-content-between align-items-center">
          <h5 className="mb-0">এলাকাভিত্তিক বিস্তারিত রিপোর্ট</h5>
          <div>
            <Button variant="outline-success" onClick={exportToExcel} className="me-2">
              <FaDownload /> Excel
            </Button>
            <Button variant="outline-primary" onClick={exportToPDF}>
              <FaPrint /> PDF
            </Button>
          </div>
        </Card.Header>
        <Card.Body className="p-0">
          <div className="table-responsive">
            <Table striped hover className="mb-0">
              <thead className="table-dark">
                <tr>
                  <th>এলাকা</th>
                  <th>মোট গ্রাহক</th>
                  <th>মোট আদায়</th>
                  <th>বকেয়া বিল</th>
                  <th>আদায়ের হার</th>
                  <th>অবস্থা</th>
                </tr>
              </thead>
              <tbody>
                {reportData.monthly?.areaWiseData?.map((area, index) => (
                  <tr key={index}>
                    <td><strong>{area.area}</strong></td>
                    <td>{area.customers}</td>
                    <td>৳{area.collection.toLocaleString()}</td>
                    <td>{area.pending}</td>
                    <td>{((area.customers - area.pending) / area.customers * 100).toFixed(1)}%</td>
                    <td>
                      {area.pending < 20 ? (
                        <Badge bg="success">ভাল</Badge>
                      ) : area.pending < 30 ? (
                        <Badge bg="warning">মধ্যম</Badge>
                      ) : (
                        <Badge bg="danger">উন্নতি প্রয়োজন</Badge>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </Table>
          </div>
        </Card.Body>
      </Card>

      {/* Performance Metrics */}
      <Row>
        <Col md={6}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">মূল পারফরম্যান্স সূচক</h5>
            </Card.Header>
            <Card.Body>
              <div className="metric-item">
                <div className="d-flex justify-content-between">
                  <span>আদায়ের হার</span>
                  <span className="fw-bold">{reportData.monthly?.collectionRate || 0}%</span>
                </div>
                <div className="progress mt-2">
                  <div 
                    className="progress-bar bg-success" 
                    style={{width: `${reportData.monthly?.collectionRate || 0}%`}}
                  ></div>
                </div>
              </div>
              
              <div className="metric-item mt-3">
                <div className="d-flex justify-content-between">
                  <span>গ্রাহক সন্তুষ্টি</span>
                  <span className="fw-bold">92%</span>
                </div>
                <div className="progress mt-2">
                  <div className="progress-bar bg-info" style={{width: '92%'}}></div>
                </div>
              </div>
              
              <div className="metric-item mt-3">
                <div className="d-flex justify-content-between">
                  <span>বিল বিতরণ</span>
                  <span className="fw-bold">98%</span>
                </div>
                <div className="progress mt-2">
                  <div className="progress-bar bg-primary" style={{width: '98%'}}></div>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
        
        <Col md={6}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">মাসিক লক্ষ্য বনাম অর্জন</h5>
            </Card.Header>
            <Card.Body>
              <div className="target-item">
                <div className="d-flex justify-content-between">
                  <span>আদায়ের লক্ষ্য</span>
                  <span>৳25,00,000</span>
                </div>
                <div className="d-flex justify-content-between">
                  <span>অর্জিত</span>
                  <span className="fw-bold text-success">৳24,50,000</span>
                </div>
                <div className="progress mt-2">
                  <div className="progress-bar bg-success" style={{width: '98%'}}></div>
                </div>
                <small className="text-muted">98% লক্ষ্য অর্জিত</small>
              </div>
              
              <div className="target-item mt-4">
                <div className="d-flex justify-content-between">
                  <span>নতুন সংযোগের লক্ষ্য</span>
                  <span>50</span>
                </div>
                <div className="d-flex justify-content-between">
                  <span>অর্জিত</span>
                  <span className="fw-bold text-primary">42</span>
                </div>
                <div className="progress mt-2">
                  <div className="progress-bar bg-primary" style={{width: '84%'}}></div>
                </div>
                <small className="text-muted">84% লক্ষ্য অর্জিত</small>
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default PalliBidyutReports;