import React, { useState } from 'react';
import { Card, Form, Button, Row, Col, InputGroup } from 'react-bootstrap';
import { FaSave, FaBarcode, FaCamera } from 'react-icons/fa';
import { useNavigate } from 'react-router-dom';
import './AddProduct.css';

const AddProduct = () => {
  const navigate = useNavigate();
  const [validated, setValidated] = useState(false);
  const [product, setProduct] = useState({
    name: '',
    barcode: '',
    category: '',
    price: '',
    cost: '',
    stock: '',
    unit: '',
    description: '',
    image: null
  });
  
  // Handle form input changes
  const handleChange = (e) => {
    const { name, value } = e.target;
    setProduct({
      ...product,
      [name]: value
    });
  };
  
  // Handle image upload
  const handleImageChange = (e) => {
    if (e.target.files && e.target.files[0]) {
      setProduct({
        ...product,
        image: URL.createObjectURL(e.target.files[0])
      });
    }
  };
  
  // Handle barcode scan
  const handleBarcodeScan = () => {
    // In a real app, this would activate a barcode scanner
    alert('Barcode scanner activated. Scan a product barcode.');
    // Mock a scanned barcode
    setTimeout(() => {
      const mockBarcode = Math.floor(Math.random() * 10000000000000).toString();
      setProduct({
        ...product,
        barcode: mockBarcode
      });
    }, 1000);
  };
  
  // Handle form submission
  const handleSubmit = (e) => {
    e.preventDefault();
    const form = e.currentTarget;
    
    if (form.checkValidity() === false) {
      e.stopPropagation();
      setValidated(true);
      return;
    }
    
    // In a real app, this would send the product data to the backend
    console.log('Submitting product:', product);
    
    // Show success message and redirect
    alert('Product added successfully!');
    navigate('/products');
  };
  
  return (
    <div className="add-product-container">
      <Card>
        <Card.Header>
          <h4 className="mb-0">Add New Product</h4>
        </Card.Header>
        <Card.Body>
          <Form noValidate validated={validated} onSubmit={handleSubmit}>
            <Row>
              {/* Left Column - Basic Info */}
              <Col md={8}>
                <Row className="mb-3">
                  <Form.Group as={Col} md={6}>
                    <Form.Label>Product Name</Form.Label>
                    <Form.Control
                      required
                      type="text"
                      name="name"
                      value={product.name}
                      onChange={handleChange}
                      placeholder="Enter product name"
                    />
                    <Form.Control.Feedback type="invalid">
                      Product name is required.
                    </Form.Control.Feedback>
                  </Form.Group>
                  
                  <Form.Group as={Col} md={6}>
                    <Form.Label>Barcode</Form.Label>
                    <InputGroup>
                      <Form.Control
                        type="text"
                        name="barcode"
                        value={product.barcode}
                        onChange={handleChange}
                        placeholder="Enter barcode"
                      />
                      <Button variant="outline-secondary" onClick={handleBarcodeScan}>
                        <FaBarcode />
                      </Button>
                    </InputGroup>
                  </Form.Group>
                </Row>
                
                <Row className="mb-3">
                  <Form.Group as={Col} md={6}>
                    <Form.Label>Category</Form.Label>
                    <Form.Select
                      required
                      name="category"
                      value={product.category}
                      onChange={handleChange}
                    >
                      <option value="">Select Category</option>
                      <option value="beverages">Beverages</option>
                      <option value="snacks">Snacks</option>
                      <option value="chocolates">Chocolates</option>
                      <option value="bakery">Bakery</option>
                      <option value="dairy">Dairy</option>
                      <option value="groceries">Groceries</option>
                      <option value="toiletries">Toiletries</option>
                    </Form.Select>
                    <Form.Control.Feedback type="invalid">
                      Please select a category.
                    </Form.Control.Feedback>
                  </Form.Group>
                  
                  <Form.Group as={Col} md={6}>
                    <Form.Label>Unit</Form.Label>
                    <Form.Select
                      name="unit"
                      value={product.unit}
                      onChange={handleChange}
                    >
                      <option value="">Select Unit</option>
                      <option value="piece">Piece</option>
                      <option value="kg">Kilogram (kg)</option>
                      <option value="g">Gram (g)</option>
                      <option value="l">Liter (l)</option>
                      <option value="ml">Milliliter (ml)</option>
                      <option value="pack">Pack</option>
                      <option value="box">Box</option>
                    </Form.Select>
                  </Form.Group>
                </Row>
                
                <Row className="mb-3">
                  <Form.Group as={Col} md={4}>
                    <Form.Label>Selling Price (৳)</Form.Label>
                    <Form.Control
                      required
                      type="number"
                      name="price"
                      value={product.price}
                      onChange={handleChange}
                      placeholder="0.00"
                      min="0"
                      step="0.01"
                    />
                    <Form.Control.Feedback type="invalid">
                      Please enter a valid price.
                    </Form.Control.Feedback>
                  </Form.Group>
                  
                  <Form.Group as={Col} md={4}>
                    <Form.Label>Cost Price (৳)</Form.Label>
                    <Form.Control
                      required
                      type="number"
                      name="cost"
                      value={product.cost}
                      onChange={handleChange}
                      placeholder="0.00"
                      min="0"
                      step="0.01"
                    />
                    <Form.Control.Feedback type="invalid">
                      Please enter a valid cost.
                    </Form.Control.Feedback>
                  </Form.Group>
                  
                  <Form.Group as={Col} md={4}>
                    <Form.Label>Initial Stock</Form.Label>
                    <Form.Control
                      required
                      type="number"
                      name="stock"
                      value={product.stock}
                      onChange={handleChange}
                      placeholder="0"
                      min="0"
                    />
                    <Form.Control.Feedback type="invalid">
                      Please enter a valid stock quantity.
                    </Form.Control.Feedback>
                  </Form.Group>
                </Row>
                
                <Form.Group className="mb-3">
                  <Form.Label>Description</Form.Label>
                  <Form.Control
                    as="textarea"
                    name="description"
                    value={product.description}
                    onChange={handleChange}
                    rows={3}
                    placeholder="Enter product description"
                  />
                </Form.Group>
              </Col>
              
              {/* Right Column - Image Upload */}
              <Col md={4}>
                <div className="image-upload-container mb-3">
                  <div className="image-preview">
                    {product.image ? (
                      <img src={product.image} alt="Product Preview" />
                    ) : (
                      <div className="no-image">
                        <FaCamera size={40} />
                        <p>No Image</p>
                      </div>
                    )}
                  </div>
                  <Form.Group>
                    <Form.Label>Product Image</Form.Label>
                    <Form.Control
                      type="file"
                      accept="image/*"
                      onChange={handleImageChange}
                    />
                    <Form.Text className="text-muted">
                      Upload a clear image of the product.
                    </Form.Text>
                  </Form.Group>
                </div>
              </Col>
            </Row>
            
            <div className="d-flex justify-content-end mt-3">
              <Button variant="secondary" className="me-2" onClick={() => navigate('/products')}>
                Cancel
              </Button>
              <Button variant="primary" type="submit">
                <FaSave className="me-1" /> Save Product
              </Button>
            </div>
          </Form>
        </Card.Body>
      </Card>
    </div>
  );
};

export default AddProduct;