import React, { useState, useEffect } from 'react';
import { Card, Table, Button, Form, InputGroup, Row, Col, Badge, Modal } from 'react-bootstrap';
import { FaPlus, FaEdit, FaTrash, FaSearch, FaBarcode, FaFileExport } from 'react-icons/fa';
import { Link } from 'react-router-dom';
import './ProductList.css';

const ProductList = () => {
  const [products, setProducts] = useState([]);
  const [filteredProducts, setFilteredProducts] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [categoryFilter, setCategoryFilter] = useState('all');
  const [categories, setCategories] = useState([]);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [productToDelete, setProductToDelete] = useState(null);
  
  // Mock data for products (in a real app, this would come from an API)
  useEffect(() => {
    // Simulating API call to fetch products
    const mockProducts = [
      { id: 1, name: 'Coca Cola', price: 35, cost: 28, category: 'beverages', barcode: '8901234567890', stock: 50, image: 'https://via.placeholder.com/50' },
      { id: 2, name: 'Lays Chips', price: 25, cost: 18, category: 'snacks', barcode: '8901234567891', stock: 30, image: 'https://via.placeholder.com/50' },
      { id: 3, name: 'Dairy Milk', price: 60, cost: 48, category: 'chocolates', barcode: '8901234567892', stock: 25, image: 'https://via.placeholder.com/50' },
      { id: 4, name: 'Bread', price: 40, cost: 30, category: 'bakery', barcode: '8901234567893', stock: 15, image: 'https://via.placeholder.com/50' },
      { id: 5, name: 'Eggs (6pcs)', price: 42, cost: 35, category: 'dairy', barcode: '8901234567894', stock: 20, image: 'https://via.placeholder.com/50' },
      { id: 6, name: 'Milk 500ml', price: 30, cost: 24, category: 'dairy', barcode: '8901234567895', stock: 18, image: 'https://via.placeholder.com/50' },
      { id: 7, name: 'Pepsi', price: 35, cost: 28, category: 'beverages', barcode: '8901234567896', stock: 45, image: 'https://via.placeholder.com/50' },
      { id: 8, name: 'Biscuits', price: 20, cost: 15, category: 'snacks', barcode: '8901234567897', stock: 40, image: 'https://via.placeholder.com/50' },
      { id: 9, name: 'Rice 1kg', price: 65, cost: 55, category: 'groceries', barcode: '8901234567898', stock: 25, image: 'https://via.placeholder.com/50' },
      { id: 10, name: 'Sugar 1kg', price: 50, cost: 42, category: 'groceries', barcode: '8901234567899', stock: 30, image: 'https://via.placeholder.com/50' },
      { id: 11, name: 'Tea 250g', price: 120, cost: 95, category: 'groceries', barcode: '8901234567810', stock: 15, image: 'https://via.placeholder.com/50' },
      { id: 12, name: 'Soap', price: 35, cost: 28, category: 'toiletries', barcode: '8901234567811', stock: 35, image: 'https://via.placeholder.com/50' },
    ];
    
    setProducts(mockProducts);
    setFilteredProducts(mockProducts);
    
    // Extract unique categories
    const uniqueCategories = [...new Set(mockProducts.map(product => product.category))];
    setCategories(uniqueCategories);
  }, []);
  
  // Filter products based on search and category
  useEffect(() => {
    let filtered = products;
    
    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(product => 
        product.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        product.barcode.includes(searchTerm)
      );
    }
    
    // Filter by category
    if (categoryFilter !== 'all') {
      filtered = filtered.filter(product => product.category === categoryFilter);
    }
    
    setFilteredProducts(filtered);
  }, [searchTerm, categoryFilter, products]);
  
  // Handle delete confirmation
  const confirmDelete = (product) => {
    setProductToDelete(product);
    setShowDeleteModal(true);
  };
  
  // Handle actual delete
  const handleDelete = () => {
    // In a real app, this would call an API to delete the product
    const updatedProducts = products.filter(p => p.id !== productToDelete.id);
    setProducts(updatedProducts);
    setShowDeleteModal(false);
    setProductToDelete(null);
  };
  
  // Handle stock status display
  const getStockStatus = (stock) => {
    if (stock <= 0) {
      return <Badge bg="danger">Out of Stock</Badge>;
    } else if (stock < 10) {
      return <Badge bg="warning">Low Stock</Badge>;
    } else {
      return <Badge bg="success">In Stock</Badge>;
    }
  };
  
  // Export products to CSV
  const exportToCSV = () => {
    // In a real app, this would generate and download a CSV file
    console.log('Exporting products to CSV...');
    alert('Products exported to CSV successfully!');
  };
  
  return (
    <div className="product-list-container">
      <Card>
        <Card.Header>
          <div className="d-flex justify-content-between align-items-center">
            <h4 className="mb-0">Product Management</h4>
            <div>
              <Button 
                as={Link} 
                to="/products/add" 
                variant="primary" 
                className="me-2"
              >
                <FaPlus className="me-1" /> Add Product
              </Button>
              <Button 
                variant="success" 
                onClick={exportToCSV}
              >
                <FaFileExport className="me-1" /> Export
              </Button>
            </div>
          </div>
        </Card.Header>
        <Card.Body>
          {/* Search and Filter */}
          <Row className="mb-3">
            <Col md={6}>
              <InputGroup>
                <InputGroup.Text>
                  <FaSearch />
                </InputGroup.Text>
                <Form.Control
                  placeholder="Search products by name or barcode..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </InputGroup>
            </Col>
            <Col md={6}>
              <Form.Select 
                value={categoryFilter}
                onChange={(e) => setCategoryFilter(e.target.value)}
              >
                <option value="all">All Categories</option>
                {categories.map(category => (
                  <option key={category} value={category}>
                    {category.charAt(0).toUpperCase() + category.slice(1)}
                  </option>
                ))}
              </Form.Select>
            </Col>
          </Row>
          
          {/* Products Table */}
          <div className="table-responsive">
            <Table striped bordered hover>
              <thead>
                <tr>
                  <th>Image</th>
                  <th>Name</th>
                  <th>Barcode</th>
                  <th>Category</th>
                  <th>Price</th>
                  <th>Cost</th>
                  <th>Stock</th>
                  <th>Status</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
                {filteredProducts.length === 0 ? (
                  <tr>
                    <td colSpan="9" className="text-center py-3">
                      No products found
                    </td>
                  </tr>
                ) : (
                  filteredProducts.map(product => (
                    <tr key={product.id}>
                      <td>
                        <img 
                          src={product.image} 
                          alt={product.name} 
                          className="product-thumbnail" 
                        />
                      </td>
                      <td>{product.name}</td>
                      <td>
                        <div className="d-flex align-items-center">
                          <span className="me-2">{product.barcode}</span>
                          <FaBarcode />
                        </div>
                      </td>
                      <td>{product.category}</td>
                      <td>৳{product.price}</td>
                      <td>৳{product.cost}</td>
                      <td>{product.stock}</td>
                      <td>{getStockStatus(product.stock)}</td>
                      <td>
                        <Button 
                          as={Link} 
                          to={`/products/edit/${product.id}`} 
                          variant="info" 
                          size="sm" 
                          className="me-1"
                        >
                          <FaEdit />
                        </Button>
                        <Button 
                          variant="danger" 
                          size="sm"
                          onClick={() => confirmDelete(product)}
                        >
                          <FaTrash />
                        </Button>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </Table>
          </div>
        </Card.Body>
      </Card>
      
      {/* Delete Confirmation Modal */}
      <Modal show={showDeleteModal} onHide={() => setShowDeleteModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Confirm Delete</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          {productToDelete && (
            <p>
              Are you sure you want to delete the product <strong>{productToDelete.name}</strong>?
              This action cannot be undone.
            </p>
          )}
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowDeleteModal(false)}>
            Cancel
          </Button>
          <Button variant="danger" onClick={handleDelete}>
            Delete
          </Button>
        </Modal.Footer>
      </Modal>
    </div>
  );
};

export default ProductList;