import React, { useState } from 'react';
import { Modal, Button, Form, Row, Col, Table } from 'react-bootstrap';
import { FaCreditCard, FaMoneyBillWave, FaMobileAlt, FaReceipt } from 'react-icons/fa';
import './Checkout.css';

const Checkout = ({ show, onHide, cart, total, onCheckout }) => {
  const [paymentMethod, setPaymentMethod] = useState('cash');
  const [paymentDetails, setPaymentDetails] = useState({
    cashAmount: '',
    cardNumber: '',
    cardType: 'visa',
    mobileNumber: '',
    provider: 'bkash',
    transactionId: ''
  });
  const [discount, setDiscount] = useState(0);
  const [tax, setTax] = useState(0);
  
  // Calculate grand total
  const grandTotal = (total - discount) * (1 + tax / 100);
  
  // Calculate change amount for cash payments
  const changeAmount = paymentMethod === 'cash' && paymentDetails.cashAmount 
    ? parseFloat(paymentDetails.cashAmount) - grandTotal 
    : 0;
  
  const handlePaymentMethodChange = (method) => {
    setPaymentMethod(method);
  };
  
  const handlePaymentDetailChange = (e) => {
    const { name, value } = e.target;
    setPaymentDetails({
      ...paymentDetails,
      [name]: value
    });
  };
  
  const handleSubmit = (e) => {
    e.preventDefault();
    
    // Validate payment details
    if (paymentMethod === 'cash' && (!paymentDetails.cashAmount || parseFloat(paymentDetails.cashAmount) < grandTotal)) {
      alert('Cash amount must be greater than or equal to the total amount');
      return;
    }
    
    if (paymentMethod === 'card' && (!paymentDetails.cardNumber || paymentDetails.cardNumber.length < 16)) {
      alert('Please enter a valid card number');
      return;
    }
    
    if (paymentMethod === 'mobile' && (!paymentDetails.mobileNumber || !paymentDetails.transactionId)) {
      alert('Please enter mobile number and transaction ID');
      return;
    }
    
    // Process checkout
    onCheckout({
      items: cart,
      total,
      discount,
      tax,
      grandTotal,
      paymentMethod,
      paymentDetails: {
        ...paymentDetails,
        changeAmount: paymentMethod === 'cash' ? changeAmount : 0
      }
    });
  };
  
  return (
    <Modal show={show} onHide={onHide} size="lg" centered>
      <Modal.Header closeButton>
        <Modal.Title>Checkout</Modal.Title>
      </Modal.Header>
      <Modal.Body>
        <Form onSubmit={handleSubmit}>
          {/* Cart Summary */}
          <div className="cart-summary mb-4">
            <h5>Order Summary</h5>
            <Table striped bordered hover size="sm">
              <thead>
                <tr>
                  <th>Product</th>
                  <th>Qty</th>
                  <th>Price</th>
                  <th>Total</th>
                </tr>
              </thead>
              <tbody>
                {cart.map((item) => (
                  <tr key={item.id}>
                    <td>{item.name}</td>
                    <td>{item.quantity}</td>
                    <td>৳{item.price.toFixed(2)}</td>
                    <td>৳{(item.price * item.quantity).toFixed(2)}</td>
                  </tr>
                ))}
              </tbody>
            </Table>
          </div>
          
          {/* Payment Method Selection */}
          <div className="payment-methods mb-4">
            <h5>Payment Method</h5>
            <div className="payment-method-buttons">
              <Button 
                variant={paymentMethod === 'cash' ? 'primary' : 'outline-primary'} 
                onClick={() => handlePaymentMethodChange('cash')}
                className="me-2 mb-2"
              >
                <FaMoneyBillWave /> Cash
              </Button>
              <Button 
                variant={paymentMethod === 'card' ? 'primary' : 'outline-primary'} 
                onClick={() => handlePaymentMethodChange('card')}
                className="me-2 mb-2"
              >
                <FaCreditCard /> Card
              </Button>
              <Button 
                variant={paymentMethod === 'mobile' ? 'primary' : 'outline-primary'} 
                onClick={() => handlePaymentMethodChange('mobile')}
                className="mb-2"
              >
                <FaMobileAlt /> Mobile Banking
              </Button>
            </div>
          </div>
          
          {/* Payment Details */}
          <div className="payment-details mb-4">
            <h5>Payment Details</h5>
            
            {paymentMethod === 'cash' && (
              <Row>
                <Form.Group as={Col} md={6} className="mb-3">
                  <Form.Label>Cash Amount</Form.Label>
                  <Form.Control
                    type="number"
                    name="cashAmount"
                    value={paymentDetails.cashAmount}
                    onChange={handlePaymentDetailChange}
                    placeholder="Enter cash amount"
                    min={grandTotal}
                    step="0.01"
                    required
                  />
                </Form.Group>
                <Form.Group as={Col} md={6} className="mb-3">
                  <Form.Label>Change</Form.Label>
                  <Form.Control
                    type="text"
                    value={changeAmount > 0 ? `৳${changeAmount.toFixed(2)}` : '৳0.00'}
                    readOnly
                    className="bg-light"
                  />
                </Form.Group>
              </Row>
            )}
            
            {paymentMethod === 'card' && (
              <Row>
                <Form.Group as={Col} md={6} className="mb-3">
                  <Form.Label>Card Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="cardNumber"
                    value={paymentDetails.cardNumber}
                    onChange={handlePaymentDetailChange}
                    placeholder="Enter card number"
                    required
                  />
                </Form.Group>
                <Form.Group as={Col} md={6} className="mb-3">
                  <Form.Label>Card Type</Form.Label>
                  <Form.Select
                    name="cardType"
                    value={paymentDetails.cardType}
                    onChange={handlePaymentDetailChange}
                  >
                    <option value="visa">Visa</option>
                    <option value="mastercard">Mastercard</option>
                    <option value="amex">American Express</option>
                    <option value="other">Other</option>
                  </Form.Select>
                </Form.Group>
              </Row>
            )}
            
            {paymentMethod === 'mobile' && (
              <Row>
                <Form.Group as={Col} md={4} className="mb-3">
                  <Form.Label>Provider</Form.Label>
                  <Form.Select
                    name="provider"
                    value={paymentDetails.provider}
                    onChange={handlePaymentDetailChange}
                  >
                    <option value="bkash">bKash</option>
                    <option value="nagad">Nagad</option>
                    <option value="rocket">Rocket</option>
                    <option value="upay">Upay</option>
                  </Form.Select>
                </Form.Group>
                <Form.Group as={Col} md={4} className="mb-3">
                  <Form.Label>Mobile Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="mobileNumber"
                    value={paymentDetails.mobileNumber}
                    onChange={handlePaymentDetailChange}
                    placeholder="Enter mobile number"
                    required
                  />
                </Form.Group>
                <Form.Group as={Col} md={4} className="mb-3">
                  <Form.Label>Transaction ID</Form.Label>
                  <Form.Control
                    type="text"
                    name="transactionId"
                    value={paymentDetails.transactionId}
                    onChange={handlePaymentDetailChange}
                    placeholder="Enter transaction ID"
                    required
                  />
                </Form.Group>
              </Row>
            )}
          </div>
          
          {/* Discount and Tax */}
          <Row className="mb-4">
            <Form.Group as={Col} md={6} className="mb-3">
              <Form.Label>Discount (৳)</Form.Label>
              <Form.Control
                type="number"
                value={discount}
                onChange={(e) => setDiscount(parseFloat(e.target.value) || 0)}
                placeholder="Enter discount amount"
                min="0"
                step="0.01"
              />
            </Form.Group>
            <Form.Group as={Col} md={6} className="mb-3">
              <Form.Label>Tax (%)</Form.Label>
              <Form.Control
                type="number"
                value={tax}
                onChange={(e) => setTax(parseFloat(e.target.value) || 0)}
                placeholder="Enter tax percentage"
                min="0"
                step="0.01"
              />
            </Form.Group>
          </Row>
          
          {/* Total Calculation */}
          <div className="totals-section">
            <Row>
              <Col xs={8} className="text-end">
                <p>Subtotal:</p>
                <p>Discount:</p>
                <p>Tax ({tax}%):</p>
                <h5>Grand Total:</h5>
              </Col>
              <Col xs={4} className="text-end">
                <p>৳{total.toFixed(2)}</p>
                <p>৳{discount.toFixed(2)}</p>
                <p>৳{((total - discount) * (tax / 100)).toFixed(2)}</p>
                <h5>৳{grandTotal.toFixed(2)}</h5>
              </Col>
            </Row>
          </div>
          
          <div className="d-flex justify-content-end mt-4">
            <Button variant="secondary" onClick={onHide} className="me-2">
              Cancel
            </Button>
            <Button variant="success" type="submit">
              <FaReceipt className="me-1" /> Complete Sale
            </Button>
          </div>
        </Form>
      </Modal.Body>
    </Modal>
  );
};

export default Checkout;