import React, { useState, useEffect } from 'react';
import { Container, Card, Table, Button, Row, Col } from 'react-bootstrap';
import { useParams } from 'react-router-dom';
import axios from 'axios';

const Invoice = () => {
  const { id } = useParams();
  const [sale, setSale] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchSale();
  }, [id]);

  const fetchSale = async () => {
    try {
      const response = await axios.get(`http://localhost:5000/api/sales/${id}`);
      setSale(response.data);
    } catch (error) {
      console.error('Error fetching sale:', error);
    } finally {
      setLoading(false);
    }
  };

  const handlePrint = () => {
    window.print();
  };

  if (loading) {
    return (
      <Container className="p-4">
        <div className="text-center">Loading invoice...</div>
      </Container>
    );
  }

  if (!sale) {
    return (
      <Container className="p-4">
        <div className="text-center">Invoice not found</div>
      </Container>
    );
  }

  return (
    <Container className="p-4">
      <div className="d-flex justify-content-between align-items-center mb-4">
        <h2>Invoice #{sale.receiptNumber || sale.id}</h2>
        <Button variant="primary" onClick={handlePrint}>
          Print Invoice
        </Button>
      </div>

      <Card>
        <Card.Body>
          <Row className="mb-4">
            <Col md={6}>
              <h5>POS System</h5>
              <p>
                123 Business Street<br />
                City, State 12345<br />
                Phone: (555) 123-4567
              </p>
            </Col>
            <Col md={6} className="text-end">
              <p>
                <strong>Date:</strong> {new Date(sale.createdAt).toLocaleDateString()}<br />
                <strong>Receipt #:</strong> {sale.receiptNumber || sale.id}<br />
                <strong>Customer:</strong> {sale.customer?.name || 'Walk-in Customer'}
              </p>
            </Col>
          </Row>

          <Table striped>
            <thead>
              <tr>
                <th>Item</th>
                <th>Qty</th>
                <th>Price</th>
                <th>Total</th>
              </tr>
            </thead>
            <tbody>
              {sale.saleItems?.map((item, index) => (
                <tr key={index}>
                  <td>{item.product?.name || 'Product'}</td>
                  <td>{item.quantity}</td>
                  <td>${parseFloat(item.unitPrice).toFixed(2)}</td>
                  <td>${parseFloat(item.totalPrice).toFixed(2)}</td>
                </tr>
              )) || (
                <tr>
                  <td colSpan="4" className="text-center">No items found</td>
                </tr>
              )}
            </tbody>
          </Table>

          <Row className="mt-4">
            <Col md={6}></Col>
            <Col md={6}>
              <Table borderless>
                <tbody>
                  <tr>
                    <td><strong>Subtotal:</strong></td>
                    <td className="text-end">${parseFloat(sale.totalAmount - (sale.tax || 0)).toFixed(2)}</td>
                  </tr>
                  <tr>
                    <td><strong>Tax:</strong></td>
                    <td className="text-end">${parseFloat(sale.tax || 0).toFixed(2)}</td>
                  </tr>
                  <tr>
                    <td><strong>Discount:</strong></td>
                    <td className="text-end">-${parseFloat(sale.discount || 0).toFixed(2)}</td>
                  </tr>
                  <tr className="border-top">
                    <td><strong>Total:</strong></td>
                    <td className="text-end"><strong>${parseFloat(sale.totalAmount).toFixed(2)}</strong></td>
                  </tr>
                  <tr>
                    <td><strong>Payment Method:</strong></td>
                    <td className="text-end text-capitalize">{sale.paymentMethod}</td>
                  </tr>
                </tbody>
              </Table>
            </Col>
          </Row>

          <div className="text-center mt-4">
            <p>Thank you for your business!</p>
          </div>
        </Card.Body>
      </Card>
    </Container>
  );
};

export default Invoice;