import Tesseract from 'tesseract.js';

class OCRService {
  constructor() {
    this.worker = null;
    this.isInitialized = false;
  }

  async initialize() {
    if (this.isInitialized) return;
    
    try {
      this.worker = await Tesseract.createWorker('eng');
      await this.worker.setParameters({
        tessedit_char_whitelist: '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz.,: ',
      });
      this.isInitialized = true;
      console.log('OCR Worker initialized successfully');
    } catch (error) {
      console.error('Failed to initialize OCR worker:', error);
      throw error;
    }
  }

  async extractTextFromImage(imageFile) {
    if (!this.isInitialized) {
      await this.initialize();
    }

    try {
      const { data: { text, confidence } } = await this.worker.recognize(imageFile);
      return {
        text: text.trim(),
        confidence: Math.round(confidence * 100) / 100
      };
    } catch (error) {
      console.error('OCR extraction failed:', error);
      throw error;
    }
  }

  extractTransactionData(ocrText) {
    const cleanText = ocrText.replace(/\s+/g, ' ').trim();
    console.log('Processing OCR text:', cleanText);

    // Regex patterns for different bKash transaction fields
    const patterns = {
      // Customer number patterns (11-digit mobile numbers starting with 01)
      customerNumber: [
        /(?:Customer|Mobile|Number|To|From)[\s:]*(\+?88)?0?1[3-9]\d{8}/i,
        /\b0?1[3-9]\d{8}\b/g,
        /(?:01[3-9]\d{8})/g
      ],
      
      // Transaction ID patterns (alphanumeric codes)
      transactionId: [
        /(?:Transaction|TrxID|ID|Reference)[\s:]*([A-Z0-9]{8,15})/i,
        /\b[A-Z]{2,4}\d{4,8}[A-Z]{0,4}\b/g,
        /\b[A-Z0-9]{8,15}\b/g
      ],
      
      // Amount patterns (decimal numbers with Tk or BDT)
      amount: [
        /(?:Amount|Tk|BDT|Taka)[\s:]*(\d{1,6}(?:\.\d{2})?)/i,
        /\b(\d{1,6}\.\d{2})\s*(?:Tk|BDT|Taka)/i,
        /(?:Cash In|Cash Out|Recharge)[\s:]*(\d{1,6}(?:\.\d{2})?)/i
      ],
      
      // Balance patterns
      oldBalance: [
        /(?:Old|Previous|Before)[\s]*(?:Balance|Bal)[\s:]*(\d{1,8}(?:\.\d{2})?)/i,
        /(?:Balance|Bal)[\s]*(?:Before|Old)[\s:]*(\d{1,8}(?:\.\d{2})?)/i,
        /\b(\d{1,8}\.\d{2})\s*(?:Old|Previous)/i
      ],
      
      newBalance: [
        /(?:New|Current|After)[\s]*(?:Balance|Bal)[\s:]*(\d{1,8}(?:\.\d{2})?)/i,
        /(?:Balance|Bal)[\s]*(?:After|New|Current)[\s:]*(\d{1,8}(?:\.\d{2})?)/i,
        /\b(\d{1,8}\.\d{2})\s*(?:New|Current)/i
      ],
      
      // Commission patterns
      commission: [
        /(?:Commission|Charge|Fee)[\s:]*(\d{1,4}(?:\.\d{2})?)/i,
        /(?:Earned|Commission)[\s:]*(\d{1,4}(?:\.\d{2})?)/i,
        /\b(\d{1,4}\.\d{2})\s*(?:Commission|Charge|Fee)/i
      ]
    };

    const extractedData = {};

    // Extract each field using multiple patterns
    for (const [field, fieldPatterns] of Object.entries(patterns)) {
      let value = null;
      
      for (const pattern of fieldPatterns) {
        const matches = cleanText.match(pattern);
        if (matches) {
          if (field === 'customerNumber') {
            // For customer number, find the first valid 11-digit number
            const numbers = cleanText.match(/\b0?1[3-9]\d{8}\b/g);
            if (numbers && numbers.length > 0) {
              value = numbers[0].startsWith('0') ? numbers[0] : '0' + numbers[0];
              break;
            }
          } else if (field === 'transactionId') {
            // For transaction ID, find alphanumeric codes
            const ids = cleanText.match(/\b[A-Z]{2,4}\d{4,8}[A-Z]{0,4}\b/g);
            if (ids && ids.length > 0) {
              value = ids[0];
              break;
            }
          } else {
            // For numeric fields, extract the first captured group
            value = matches[1] || matches[0];
            if (value) {
              value = parseFloat(value.replace(/[^\d.]/g, ''));
              if (!isNaN(value)) break;
            }
          }
        }
      }
      
      extractedData[field] = value;
    }

    // Determine transaction type based on keywords
    let transactionType = 'mobile_recharge'; // default
    if (cleanText.toLowerCase().includes('cash in')) {
      transactionType = 'cash_in';
    } else if (cleanText.toLowerCase().includes('cash out')) {
      transactionType = 'cash_out';
    }
    
    extractedData.transactionType = transactionType;

    // Validate extracted data
    const validation = this.validateExtractedData(extractedData);
    
    return {
      ...extractedData,
      validation,
      rawText: ocrText
    };
  }

  validateExtractedData(data) {
    const errors = [];
    const warnings = [];

    // Validate customer number
    if (!data.customerNumber) {
      errors.push('Customer number not found');
    } else if (!/^01[3-9]\d{8}$/.test(data.customerNumber)) {
      errors.push('Invalid customer number format');
    }

    // Validate transaction ID
    if (!data.transactionId) {
      errors.push('Transaction ID not found');
    } else if (data.transactionId.length < 6) {
      warnings.push('Transaction ID seems too short');
    }

    // Validate amount
    if (!data.amount || data.amount <= 0) {
      errors.push('Invalid or missing amount');
    }

    // Validate balances
    if (!data.oldBalance && data.oldBalance !== 0) {
      warnings.push('Old balance not found');
    }
    if (!data.newBalance && data.newBalance !== 0) {
      warnings.push('New balance not found');
    }

    // Validate commission
    if (!data.commission && data.commission !== 0) {
      warnings.push('Commission not found');
    }

    // Cross-validation: check if balance calculation makes sense
    if (data.oldBalance !== null && data.newBalance !== null && data.amount !== null) {
      const expectedNewBalance = data.transactionType === 'cash_out' 
        ? data.oldBalance - data.amount 
        : data.oldBalance + data.amount;
      
      const balanceDiff = Math.abs(data.newBalance - expectedNewBalance);
      if (balanceDiff > 1) { // Allow small rounding differences
        warnings.push('Balance calculation doesn\'t match transaction amount');
      }
    }

    return {
      isValid: errors.length === 0,
      errors,
      warnings,
      confidence: errors.length === 0 ? (warnings.length === 0 ? 'high' : 'medium') : 'low'
    };
  }

  async cleanup() {
    if (this.worker) {
      await this.worker.terminate();
      this.worker = null;
      this.isInitialized = false;
    }
  }

  async processOCRImage(imageFile) {
    try {
      // Extract text from image
      const { text, confidence } = await this.extractTextFromImage(imageFile);
      
      // Extract transaction data from text
      const extractedData = this.extractTransactionData(text);
      
      // Validate the extracted data
      const validation = this.validateExtractedData(extractedData);
      
      return {
        rawText: text,
        extractedData: {
          ...extractedData,
          confidence: confidence
        },
        validation,
        confidence: confidence
      };
    } catch (error) {
      console.error('Error processing OCR image:', error);
      throw error;
    }
  }
}

const ocrService = new OCRService();

// Named export for the processOCRImage function
export const processOCRImage = (imageFile) => ocrService.processOCRImage(imageFile);

export default ocrService;